<?php
// Start session dan validasi login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Validasi hak akses (hanya admin dan owner)
$allowed_roles = ['1', '4']; // Sesuaikan dengan id_role admin dan owner
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

include 'header.php';
require_once 'config.php';

// Ambil ID pengguna dari parameter URL
$id_pengguna = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Query untuk mendapatkan detail pengguna
$query = "SELECT p.*, t.nama_toko, r.nama_role 
          FROM pengguna p
          LEFT JOIN toko t ON p.id_toko = t.id_toko
          JOIN role r ON p.id_role = r.id_role
          WHERE p.id_pengguna = ?";
$stmt = mysqli_prepare($conn, $query);
mysqli_stmt_bind_param($stmt, "i", $id_pengguna);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$pengguna = mysqli_fetch_assoc($result);

// Jika data tidak ditemukan
if (!$pengguna) {
    include '404.php';
    exit();
}

// Format tanggal terakhir login
$last_login = $pengguna['last_login'] ? date('d M Y H:i', strtotime($pengguna['last_login'])) : 'Belum pernah login';
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Detail Pengguna</h1>
        <div>
            <a href="pengguna.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                <i class="fas fa-arrow-left fa-sm text-white-50"></i> Kembali
            </a>
            <a href="edit_pengguna.php?id=<?php echo $id_pengguna; ?>" class="d-none d-sm-inline-block btn btn-sm btn-primary-custom shadow-sm">
                <i class="fas fa-edit fa-sm text-white-50"></i> Edit
            </a>
        </div>
    </div>

    <!-- Card untuk detail pengguna -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 bg-primary-custom">
            <h6 class="m-0 font-weight-bold text-white">Informasi Pengguna</h6>
        </div>
        <div class="card-body">
            <div class="row">
                <!-- Kolom kiri - Foto profil dan info dasar -->
                <div class="col-md-4 text-center">
                    <div class="mb-4">
                        <?php if ($pengguna['foto_profil']): ?>
                            <img src="<?php echo htmlspecialchars($pengguna['foto_profil']); ?>" class="img-fluid rounded-circle" style="width: 200px; height: 200px; object-fit: cover;" alt="Foto Profil">
                        <?php else: ?>
                            <div class="rounded-circle bg-secondary d-flex align-items-center justify-content-center" style="width: 200px; height: 200px; margin: 0 auto;">
                                <i class="fas fa-user text-white" style="font-size: 80px;"></i>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card mb-4">
                        <div class="card-body text-center">
                            <h5 class="card-title"><?php echo htmlspecialchars($pengguna['nama_lengkap']); ?></h5>
                            <p class="card-text">
                                <span class="badge badge-<?php echo $pengguna['is_active'] ? 'success' : 'danger'; ?>">
                                    <?php echo $pengguna['is_active'] ? 'Aktif' : 'Nonaktif'; ?>
                                </span>
                            </p>
                            <p class="card-text text-muted"><?php echo htmlspecialchars($pengguna['nama_role']); ?></p>
                        </div>
                    </div>
                </div>
                
                <!-- Kolom kanan - Detail lengkap -->
                <div class="col-md-8">
                    <div class="card mb-4">
                        <div class="card-header bg-light">
                            <h6 class="m-0 font-weight-bold text-primary-custom">Data Pribadi</h6>
                        </div>
                        <div class="card-body">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Username</label>
                                    <p><?php echo htmlspecialchars($pengguna['username']); ?></p>
                                </div>
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Nama Lengkap</label>
                                    <p><?php echo htmlspecialchars($pengguna['nama_lengkap']); ?></p>
                                </div>
                            </div>
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Email</label>
                                    <p><?php echo $pengguna['email'] ? htmlspecialchars($pengguna['email']) : '-'; ?></p>
                                </div>
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Telepon</label>
                                    <p><?php echo $pengguna['telepon'] ? htmlspecialchars($pengguna['telepon']) : '-'; ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card mb-4">
                        <div class="card-header bg-light">
                            <h6 class="m-0 font-weight-bold text-primary-custom">Informasi Tambahan</h6>
                        </div>
                        <div class="card-body">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Toko</label>
                                    <p><?php echo $pengguna['nama_toko'] ? htmlspecialchars($pengguna['nama_toko']) : '-'; ?></p>
                                </div>
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Role</label>
                                    <p><?php echo htmlspecialchars($pengguna['nama_role']); ?></p>
                                </div>
                            </div>
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Terakhir Login</label>
                                    <p><?php echo $last_login; ?></p>
                                </div>
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Tanggal Dibuat</label>
                                    <p><?php echo date('d M Y H:i', strtotime($pengguna['created_at'])); ?></p>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <label class="font-weight-bold">Terakhir Diupdate</label>
                                    <p><?php echo date('d M Y H:i', strtotime($pengguna['updated_at'])); ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Tombol aksi -->
            <div class="d-flex justify-content-between mt-4">
                <a href="pengguna.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Kembali
                </a>
                <div>
                    <a href="edit_pengguna.php?id=<?php echo $id_pengguna; ?>" class="btn btn-primary-custom">
                        <i class="fas fa-edit"></i> Edit
                    </a>
                    <button class="btn btn-danger btn-hapus" 
                            data-id="<?php echo $id_pengguna; ?>"
                            data-nama="<?php echo htmlspecialchars($pengguna['nama_lengkap']); ?>">
                        <i class="fas fa-trash"></i> Hapus
                    </button>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Modal konfirmasi hapus -->
<div class="modal fade" id="hapusModal" tabindex="-1" role="dialog" aria-labelledby="hapusModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="hapusModalLabel">Konfirmasi Hapus Pengguna</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Anda yakin ingin menghapus pengguna <strong id="namaPenggunaHapus"></strong>?</p>
                <p class="text-danger">Aksi ini tidak dapat dibatalkan!</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Batal</button>
                <form id="formHapus" method="POST" action="delete_pengguna.php">
                    <input type="hidden" name="id_pengguna" id="idPenggunaHapus">
                    <button type="submit" class="btn btn-danger">Hapus</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Modal konfirmasi hapus
    $('.btn-hapus').click(function() {
        var id = $(this).data('id');
        var nama = $(this).data('nama');
        
        $('#namaPenggunaHapus').text(nama);
        $('#idPenggunaHapus').val(id);
        $('#hapusModal').modal('show');
    });
});
</script>

<?php include 'footer.php'; ?>