<?php
session_start();
require_once 'config.php';

// Cek koneksi database
if (!$conn) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Cek hak akses
if (!isset($_SESSION['id_role']) || !checkAccess($_SESSION['id_role'], 'penjualan', 'bisa_edit')) {
    die("Anda tidak memiliki hak akses untuk mengedit transaksi");
}

// Ambil ID penjualan dari URL
$id_penjualan = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($id_penjualan <= 0) {
    die("ID penjualan tidak valid");
}

// Query untuk mendapatkan data penjualan
$query_penjualan = "SELECT p.*, pl.nama_pelanggan, mp.nama_metode, t.nama_toko
                   FROM penjualan p
                   LEFT JOIN pelanggan pl ON p.id_pelanggan = pl.id_pelanggan
                   JOIN metode_pembayaran mp ON p.id_metode = mp.id_metode
                   JOIN toko t ON p.id_toko = t.id_toko
                   WHERE p.id_penjualan = ?";
$stmt_penjualan = mysqli_prepare($conn, $query_penjualan);
mysqli_stmt_bind_param($stmt_penjualan, "i", $id_penjualan);
mysqli_stmt_execute($stmt_penjualan);
$result_penjualan = mysqli_stmt_get_result($stmt_penjualan);

if (mysqli_num_rows($result_penjualan) == 0) {
    die("Transaksi tidak ditemukan");
}

$penjualan = mysqli_fetch_assoc($result_penjualan);

// Cek apakah transaksi sudah lunas (tidak bisa diedit jika sudah lunas)
if ($penjualan['status_pembayaran'] == 'lunas') {
    die("Transaksi yang sudah lunas tidak dapat diedit");
}

// Query untuk mendapatkan detail penjualan
$query_detail = "SELECT dp.*, b.nama_barang, b.kode_barang, s.stok_akhir as stok
                FROM detail_penjualan dp
                JOIN barang b ON dp.id_barang = b.id_barang
                JOIN stok_toko s ON dp.id_barang = s.id_barang AND s.id_toko = ?
                WHERE dp.id_penjualan = ?";
$stmt_detail = mysqli_prepare($conn, $query_detail);
mysqli_stmt_bind_param($stmt_detail, "ii", $penjualan['id_toko'], $id_penjualan);
mysqli_stmt_execute($stmt_detail);
$result_detail = mysqli_stmt_get_result($stmt_detail);

$detail_penjualan = [];
while ($row = mysqli_fetch_assoc($result_detail)) {
    $detail_penjualan[] = $row;
}

// Query untuk dropdown pelanggan
$query_pelanggan = "SELECT * FROM pelanggan ORDER BY nama_pelanggan";
$result_pelanggan = mysqli_query($conn, $query_pelanggan);

// Query untuk dropdown toko
$query_toko = "SELECT * FROM toko WHERE status_aktif = 1";
$result_toko = mysqli_query($conn, $query_toko);

// Query untuk dropdown metode pembayaran
$query_metode = "SELECT * FROM metode_pembayaran WHERE is_active = 1";
$result_metode = mysqli_query($conn, $query_metode);

// Fungsi untuk cek hak akses
function checkAccess($id_role, $modul, $akses) {
    global $conn;
    
    $query = "SELECT $akses FROM hak_akses WHERE id_role = ? AND modul = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "is", $id_role, $modul);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        return (bool)$row[$akses];
    }
    
    return false;
}

include 'header.php';
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Edit Transaksi Penjualan</h1>
        <div class="d-none d-sm-inline-block">
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                    <li class="breadcrumb-item"><a href="penjualan.php">Penjualan</a></li>
                    <li class="breadcrumb-item active" aria-current="page">Edit</li>
                </ol>
            </nav>
        </div>
    </div>
    
    <!-- Form Edit Penjualan -->
    <div class="card animate-fade-in" style="animation-delay: 0.1s;">
        <div class="card-body">
            <form id="formEditPenjualan" action="proses_edit_penjualan.php" method="POST">
                <input type="hidden" name="id_penjualan" value="<?php echo $penjualan['id_penjualan']; ?>">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="no_transaksi" class="form-label">No. Transaksi</label>
                            <input type="text" class="form-control" id="no_transaksi" name="no_transaksi" value="<?php echo $penjualan['no_transaksi']; ?>" readonly>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="tanggal_penjualan" class="form-label">Tanggal Penjualan</label>
                            <input type="datetime-local" class="form-control" id="tanggal_penjualan" name="tanggal_penjualan" value="<?php echo date('Y-m-d\TH:i', strtotime($penjualan['tanggal_penjualan'])); ?>">
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="id_toko" class="form-label">Toko</label>
                            <select class="form-select" id="id_toko" name="id_toko" required>
                                <option value="">Pilih Toko</option>
                                <?php while ($toko = mysqli_fetch_assoc($result_toko)): ?>
                                    <option value="<?php echo $toko['id_toko']; ?>" <?php echo ($toko['id_toko'] == $penjualan['id_toko']) ? 'selected' : ''; ?>>
                                        <?php echo $toko['nama_toko']; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="id_pelanggan" class="form-label">Pelanggan (Opsional)</label>
                            <select class="form-select" id="id_pelanggan" name="id_pelanggan">
                                <option value="">Pilih Pelanggan</option>
                                <?php 
                                mysqli_data_seek($result_pelanggan, 0); // Reset pointer
                                while ($pelanggan = mysqli_fetch_assoc($result_pelanggan)): ?>
                                    <option value="<?php echo $pelanggan['id_pelanggan']; ?>" <?php echo ($pelanggan['id_pelanggan'] == $penjualan['id_pelanggan']) ? 'selected' : ''; ?>>
                                        <?php echo $pelanggan['nama_pelanggan']; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="id_metode" class="form-label">Metode Pembayaran</label>
                            <select class="form-select" id="id_metode" name="id_metode" required>
                                <option value="">Pilih Metode</option>
                                <?php 
                                mysqli_data_seek($result_metode, 0); // Reset pointer
                                while ($metode = mysqli_fetch_assoc($result_metode)): ?>
                                    <option value="<?php echo $metode['id_metode']; ?>" <?php echo ($metode['id_metode'] == $penjualan['id_metode']) ? 'selected' : ''; ?>>
                                        <?php echo $metode['nama_metode']; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="catatan" class="form-label">Catatan (Opsional)</label>
                            <textarea class="form-control" id="catatan" name="catatan" rows="1"><?php echo $penjualan['catatan']; ?></textarea>
                        </div>
                    </div>
                </div>
                
                <hr>
                
                <h5 class="mb-3">Daftar Barang</h5>
                
                <div class="row mb-3">
                    <div class="col-md-8">
                        <label for="cari_barang" class="form-label">Cari Barang</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="cari_barang" placeholder="Kode atau nama barang">
                            <button class="btn btn-outline-secondary" type="button" id="btnCariBarang">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                        <div id="hasilPencarianBarang" class="position-absolute w-100 bg-white z-index-1 d-none" style="max-height: 200px; overflow-y: auto;">
                            <!-- Hasil pencarian akan muncul di sini -->
                        </div>
                    </div>
                    <div class="col-md-4">
                        <label for="jumlah" class="form-label">Jumlah</label>
                        <input type="number" class="form-control" id="jumlah" min="1" value="1">
                    </div>
                </div>
                
                <div class="table-responsive">
                    <table class="table table-bordered" id="tabelBarang">
                        <thead>
                            <tr>
                                <th width="5%">No</th>
                                <th width="35%">Nama Barang</th>
                                <th width="15%">Harga Jual</th>
                                <th width="10%">Jumlah</th>
                                <th width="15%">Diskon</th>
                                <th width="20%">Subtotal</th>
                                <th width="5%">Aksi</th>
                            </tr>
                        </thead>
                        <tbody id="daftarBarang">
                            <?php if (empty($detail_penjualan)): ?>
                                <tr id="barangKosong">
                                    <td colspan="7" class="text-center">Belum ada barang ditambahkan</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($detail_penjualan as $index => $detail): ?>
                                    <tr id="barang-<?php echo $index + 1; ?>">
                                        <td><?php echo $index + 1; ?></td>
                                        <td>
                                            <?php echo $detail['nama_barang']; ?>
                                            <input type="hidden" name="id_barang[]" value="<?php echo $detail['id_barang']; ?>">
                                            <input type="hidden" name="harga_jual[]" value="<?php echo $detail['harga_jual']; ?>">
                                        </td>
                                        <td><?php echo format_rupiah($detail['harga_jual']); ?></td>
                                        <td>
                                            <input type="number" class="form-control form-control-sm jumlah" name="jumlah[]" min="1" value="<?php echo $detail['jumlah']; ?>" data-row="barang-<?php echo $index + 1; ?>">
                                        </td>
                                        <td>
                                            <input type="number" class="form-control form-control-sm diskon" name="diskon_item[]" min="0" value="<?php echo $detail['diskon_item']; ?>" data-row="barang-<?php echo $index + 1; ?>">
                                        </td>
                                        <td class="subtotal"><?php echo format_rupiah($detail['subtotal']); ?></td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-danger hapus-barang" data-row="barang-<?php echo $index + 1; ?>">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="5" class="text-end"><strong>Total</strong></td>
                                <td id="totalPenjualan"><?php echo format_rupiah($penjualan['total_penjualan']); ?></td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>Diskon</strong></td>
                                <td>
                                    <input type="number" class="form-control form-control-sm" id="diskon" name="diskon" min="0" value="<?php echo $penjualan['diskon']; ?>">
                                </td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>PPN</strong></td>
                                <td>
                                    <input type="number" class="form-control form-control-sm" id="ppn" name="ppn" min="0" value="<?php echo $penjualan['ppn']; ?>">
                                </td>
                                <td></td>
                            </tr>
                            <tr class="table-active">
                                <td colspan="5" class="text-end"><strong>Grand Total</strong></td>
                                <td id="grandTotal"><?php echo format_rupiah($penjualan['grand_total']); ?></td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>Jumlah Bayar</strong></td>
                                <td>
                                    <input type="number" class="form-control form-control-sm" id="jumlahBayar" name="jumlah_bayar" min="0" value="<?php echo $penjualan['jumlah_bayar']; ?>" required>
                                </td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>Kembalian</strong></td>
                                <td id="kembalian"><?php echo format_rupiah($penjualan['kembalian']); ?></td>
                                <td></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                
                <input type="hidden" id="statusPembayaran" name="status_pembayaran" value="<?php echo $penjualan['status_pembayaran']; ?>">
                <input type="hidden" id="sisaHutang" name="sisa_hutang" value="<?php echo $penjualan['sisa_hutang']; ?>">
                
                <div class="d-flex justify-content-between mt-3">
                    <a href="penjualan.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-1"></i> Kembali
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save me-1"></i> Simpan Perubahan
                    </button>
                </div>
            </form>
        </div>
    </div>
</main>

<!-- Script untuk edit penjualan -->
<script>
$(document).ready(function() {
    // Variabel untuk menyimpan data barang
    let daftarBarang = <?php echo json_encode($detail_penjualan); ?>;
    let counterBarang = <?php echo count($detail_penjualan) + 1; ?>;
    
    // Fungsi untuk menghitung total, grand total, kembalian, dll
    function hitungTotal() {
        let total = 0;
        
        // Hitung total dari daftar barang
        daftarBarang.forEach(barang => {
            total += barang.subtotal;
        });
        
        // Update total penjualan
        $('#totalPenjualan').text(formatRupiah(total));
        
        // Hitung diskon dan ppn
        const diskon = parseFloat($('#diskon').val()) || 0;
        const ppn = parseFloat($('#ppn').val()) || 0;
        
        // Hitung grand total
        const grandTotal = total - diskon + ppn;
        $('#grandTotal').text(formatRupiah(grandTotal));
        
        // Hitung kembalian
        const jumlahBayar = parseFloat($('#jumlahBayar').val()) || 0;
        const kembalian = jumlahBayar - grandTotal;
        
        if (kembalian >= 0) {
            $('#kembalian').text(formatRupiah(kembalian));
            $('#statusPembayaran').val('lunas');
            $('#sisaHutang').val('0');
        } else {
            $('#kembalian').text(formatRupiah(0));
            $('#statusPembayaran').val('hutang');
            $('#sisaHutang').val(Math.abs(kembalian).toFixed(2));
        }
    }
    
    // Fungsi untuk format rupiah
    function formatRupiah(angka) {
        return 'Rp ' + angka.toLocaleString('id-ID');
    }
    
    // Fungsi untuk menambahkan barang ke tabel
    function tambahBarangKeTabel(barang) {
        // Hilangkan pesan barang kosong jika ada
        $('#barangKosong').remove();
        
        // Tambahkan row baru ke tabel
        const rowId = 'barang-' + counterBarang;
        const newRow = `
            <tr id="${rowId}">
                <td>${counterBarang}</td>
                <td>
                    ${barang.nama_barang}
                    <input type="hidden" name="id_barang[]" value="${barang.id_barang}">
                    <input type="hidden" name="harga_jual[]" value="${barang.harga_jual}">
                </td>
                <td>${formatRupiah(barang.harga_jual)}</td>
                <td>
                    <input type="number" class="form-control form-control-sm jumlah" name="jumlah[]" min="1" value="${barang.jumlah}" data-row="${rowId}">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm diskon" name="diskon_item[]" min="0" value="${barang.diskon_item}" data-row="${rowId}">
                </td>
                <td class="subtotal">${formatRupiah(barang.subtotal)}</td>
                <td>
                    <button type="button" class="btn btn-sm btn-danger hapus-barang" data-row="${rowId}">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
        
        $('#daftarBarang').append(newRow);
        counterBarang++;
    }
    
    // Event listener untuk pencarian barang
    $('#btnCariBarang').click(function() {
        const keyword = $('#cari_barang').val().trim();
        
        if (keyword.length < 2) {
            alert('Masukkan minimal 2 karakter untuk pencarian');
            return;
        }
        
        $.ajax({
            url: 'ajax_cari_barang.php',
            type: 'GET',
            data: { keyword: keyword },
            dataType: 'json',
            success: function(response) {
                const hasilPencarian = $('#hasilPencarianBarang');
                hasilPencarian.empty();
                
                if (response.length > 0) {
                    response.forEach(barang => {
                        // Cek apakah barang sudah ada di daftar
                        const sudahAda = daftarBarang.some(item => item.id_barang == barang.id_barang);
                        
                        if (!sudahAda) {
                            hasilPencarian.append(`
                                <div class="list-group-item list-group-item-action p-2" 
                                     data-id="${barang.id_barang}" 
                                     data-nama="${barang.nama_barang}" 
                                     data-harga="${barang.harga_jual}">
                                    ${barang.nama_barang} - ${formatRupiah(barang.harga_jual)}
                                    ${barang.stok ? '<span class="badge bg-info float-end">Stok: ' + barang.stok + '</span>' : ''}
                                </div>
                            `);
                        }
                    });
                    
                    if (hasilPencarian.children().length === 0) {
                        hasilPencarian.append('<div class="list-group-item p-2">Semua barang sudah ditambahkan</div>');
                    }
                    
                    hasilPencarian.removeClass('d-none');
                } else {
                    hasilPencarian.append('<div class="list-group-item p-2">Barang tidak ditemukan</div>');
                    hasilPencarian.removeClass('d-none');
                }
            },
            error: function() {
                alert('Terjadi kesalahan saat mencari barang');
            }
        });
    });
    
    // Event listener untuk memilih barang dari hasil pencarian
    $(document).on('click', '#hasilPencarianBarang .list-group-item', function() {
        const idBarang = $(this).data('id');
        const namaBarang = $(this).data('nama');
        const hargaJual = parseFloat($(this).data('harga'));
        const jumlah = parseInt($('#jumlah').val()) || 1;
        
        // Tambahkan barang ke array daftarBarang
        const barang = {
            id_barang: idBarang,
            nama_barang: namaBarang,
            harga_jual: hargaJual,
            jumlah: jumlah,
            diskon_item: 0,
            subtotal: hargaJual * jumlah
        };
        
        daftarBarang.push(barang);
        
        // Tambahkan barang ke tabel
        tambahBarangKeTabel(barang);
        
        // Hitung ulang total
        hitungTotal();
        
        // Reset form pencarian
        $('#cari_barang').val('');
        $('#jumlah').val('1');
        $('#hasilPencarianBarang').addClass('d-none').empty();
    });
    
    // Event listener untuk menghapus barang dari tabel
    $(document).on('click', '.hapus-barang', function() {
        const rowId = $(this).data('row');
        
        // Hapus barang dari array daftarBarang
        const index = daftarBarang.findIndex(item => 'barang-' + (daftarBarang.indexOf(item) + 1) === rowId);
        if (index !== -1) {
            daftarBarang.splice(index, 1);
        }
        
        // Hapus row dari tabel
        $('#' + rowId).remove();
        
        // Jika tidak ada barang, tampilkan pesan
        if (daftarBarang.length === 0) {
            $('#daftarBarang').html('<tr id="barangKosong"><td colspan="7" class="text-center">Belum ada barang ditambahkan</td></tr>');
        } else {
            // Update nomor urut
            $('#daftarBarang tr').each(function(index) {
                $(this).find('td:first').text(index + 1);
            });
        }
        
        // Hitung ulang total
        hitungTotal();
    });
    
    // Event listener untuk perubahan jumlah atau diskon barang
    $(document).on('change', '.jumlah, .diskon', function() {
        const rowId = $(this).data('row');
        const index = parseInt(rowId.split('-')[1]) - 1;
        
        if (index >= 0 && index < daftarBarang.length) {
            const jumlah = parseFloat($(this).closest('tr').find('.jumlah').val()) || 0;
            const diskon = parseFloat($(this).closest('tr').find('.diskon').val()) || 0;
            const hargaJual = daftarBarang[index].harga_jual;
            
            // Update data di array
            daftarBarang[index].jumlah = jumlah;
            daftarBarang[index].diskon_item = diskon;
            daftarBarang[index].subtotal = (hargaJual * jumlah) - diskon;
            
            // Update subtotal di tabel
            $(this).closest('tr').find('.subtotal').text(formatRupiah(daftarBarang[index].subtotal));
            
            // Hitung ulang total
            hitungTotal();
        }
    });
    
    // Event listener untuk perubahan diskon, ppn, atau jumlah bayar
    $('#diskon, #ppn, #jumlahBayar').on('change keyup', function() {
        hitungTotal();
    });
    
    // Validasi form sebelum submit
    $('#formEditPenjualan').submit(function(e) {
        if (daftarBarang.length === 0) {
            e.preventDefault();
            alert('Minimal harus ada 1 barang dalam transaksi');
            return false;
        }
        
        const jumlahBayar = parseFloat($('#jumlahBayar').val()) || 0;
        const grandTotal = parseFloat($('#grandTotal').text().replace(/[^0-9]/g, '')) || 0;
        
        if (jumlahBayar <= 0) {
            e.preventDefault();
            alert('Jumlah bayar harus lebih dari 0');
            return false;
        }
        
        // Jika metode pembayaran adalah hutang, pastikan pelanggan dipilih
        if ($('#statusPembayaran').val() === 'hutang' && !$('#id_pelanggan').val()) {
            e.preventDefault();
            alert('Untuk transaksi hutang, pelanggan harus dipilih');
            return false;
        }
        
        return true;
    });
    
    // Tutup hasil pencarian saat klik di luar
    $(document).click(function(e) {
        if (!$(e.target).closest('#hasilPencarianBarang').length && !$(e.target).is('#cari_barang, #btnCariBarang')) {
            $('#hasilPencarianBarang').addClass('d-none');
        }
    });
});
</script>

<?php include 'footer.php'; ?>