<?php
// Start session dan validasi login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Validasi hak akses (hanya Super Admin, Admin dan Owner)
$allowed_roles = ['0', '1', '3', '4']; // Sesuaikan dengan id_role super admin, admin dan owner
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

include 'header.php';
require_once 'config.php';

// Fungsi untuk format status
function format_status($status) {
    return $status == 1 ? 'Aktif' : 'Nonaktif';
}

// Query untuk statistik pengguna
$query_statistik = "SELECT 
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as aktif,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as nonaktif,
                    COUNT(*) as total
                    FROM pengguna";
$result_statistik = mysqli_query($conn, $query_statistik);
$statistik = mysqli_fetch_assoc($result_statistik);

// Query untuk data pengguna dengan pencarian
$search = isset($_GET['search']) ? $_GET['search'] : '';
$query = "SELECT p.*, t.nama_toko, r.nama_role 
          FROM pengguna p
          LEFT JOIN toko t ON p.id_toko = t.id_toko
          JOIN role r ON p.id_role = r.id_role
          WHERE (p.username LIKE ? OR p.nama_lengkap LIKE ? OR t.nama_toko LIKE ? OR r.nama_role LIKE ?)";

// Gunakan prepared statement untuk pencarian
$stmt = mysqli_prepare($conn, $query);
$search_param = "%$search%";
mysqli_stmt_bind_param($stmt, "ssss", $search_param, $search_param, $search_param, $search_param);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

// Konfigurasi pagination
$per_page = 10;
$total_records = mysqli_num_rows($result);
$total_pages = ceil($total_records / $per_page);
$current_page = isset($_GET['page']) ? $_GET['page'] : 1;
$offset = ($current_page - 1) * $per_page;

// Query dengan pagination
$query_pagination = "SELECT p.*, t.nama_toko, r.nama_role 
                    FROM pengguna p
                    LEFT JOIN toko t ON p.id_toko = t.id_toko
                    JOIN role r ON p.id_role = r.id_role
                    WHERE (p.username LIKE ? OR p.nama_lengkap LIKE ? OR t.nama_toko LIKE ? OR r.nama_role LIKE ?)
                    ORDER BY p.nama_lengkap ASC
                    LIMIT ?, ?";
$stmt_pagination = mysqli_prepare($conn, $query_pagination);
$limit_param = $per_page;
$offset_param = $offset;
mysqli_stmt_bind_param($stmt_pagination, "ssssii", $search_param, $search_param, $search_param, $search_param, $offset_param, $limit_param);
mysqli_stmt_execute($stmt_pagination);
$result_pagination = mysqli_stmt_get_result($stmt_pagination);
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Manajemen Pengguna</h1>
        <div class="d-none d-sm-inline-block">
            <span class="text-muted">Terakhir diperbarui: <?php echo date('d M Y H:i:s'); ?></span>
        </div>
    </div>

    <!-- Statistik Ringkasan -->
    <div class="row mb-4">
        <!-- Total Pengguna -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total Pengguna</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['total']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-users fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pengguna Aktif -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Pengguna Aktif</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['aktif']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-user-check fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pengguna Nonaktif -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-danger shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                Pengguna Nonaktif</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['nonaktif']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-user-times fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Card untuk tabel -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between bg-primary-custom">
            <h6 class="m-0 font-weight-bold text-white">Daftar Pengguna</h6>
            <div>
                <a href="tambah_pengguna.php" class="btn btn-sm btn-success">
                    <i class="fas fa-plus"></i> Tambah Pengguna
                </a>
            </div>
        </div>
        <div class="card-body">
            <!-- Form Pencarian -->
            <form method="GET" action="" class="mb-4">
                <div class="input-group">
                    <input type="text" class="form-control" name="search" placeholder="Cari berdasarkan username, nama, toko, atau role..." value="<?php echo htmlspecialchars($search); ?>">
                    <button class="btn btn-primary-custom" type="submit">
                        <i class="fas fa-search"></i> Cari
                    </button>
                    <?php if (!empty($search)): ?>
                        <a href="pengguna.php" class="btn btn-danger">
                            <i class="fas fa-times"></i> Reset
                        </a>
                    <?php endif; ?>
                </div>
            </form>

            <?php if (mysqli_num_rows($result_pagination) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" id="dataTable" width="100%" cellspacing="0">
                        <thead class="bg-light">
                            <tr>
                                <th>No</th>
                                <th>Username</th>
                                <th>Nama Lengkap</th>
                                <th>Role</th>
                                <th>Toko</th>
                                <th>Email</th>
                                <th>Telepon</th>
                                <th>Status</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $no = 1 + $offset;
                            while ($row = mysqli_fetch_assoc($result_pagination)): 
                            ?>
                                <tr>
                                    <td><?php echo $no++; ?></td>
                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                    <td><?php echo htmlspecialchars($row['nama_lengkap']); ?></td>
                                    <td><?php echo htmlspecialchars($row['nama_role']); ?></td>
                                    <td><?php echo $row['nama_toko'] ? htmlspecialchars($row['nama_toko']) : '-'; ?></td>
                                    <td><?php echo $row['email'] ? htmlspecialchars($row['email']) : '-'; ?></td>
                                    <td><?php echo $row['telepon'] ? htmlspecialchars($row['telepon']) : '-'; ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $row['is_active'] ? 'success' : 'danger'; ?>">
                                            <?php echo format_status($row['is_active']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="detail_pengguna.php?id=<?php echo $row['id_pengguna']; ?>" 
                                           class="btn btn-sm btn-primary" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="edit_pengguna.php?id=<?php echo $row['id_pengguna']; ?>" 
                                           class="btn btn-sm btn-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <button class="btn btn-sm btn-danger btn-hapus" 
                                                data-id="<?php echo $row['id_pengguna']; ?>"
                                                data-nama="<?php echo htmlspecialchars($row['nama_lengkap']); ?>"
                                                title="Hapus">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <?php if ($current_page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $current_page - 1; ?>&search=<?php echo urlencode($search); ?>" aria-label="Previous">
                                    <span aria-hidden="true">&laquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($i == $current_page) ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($current_page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $current_page + 1; ?>&search=<?php echo urlencode($search); ?>" aria-label="Next">
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php else: ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Tidak ada data pengguna yang ditemukan.
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<!-- Modal konfirmasi hapus -->
<div class="modal fade" id="hapusModal" tabindex="-1" role="dialog" aria-labelledby="hapusModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="hapusModalLabel">Konfirmasi Hapus Pengguna</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Anda yakin ingin menghapus pengguna <strong id="namaPenggunaHapus"></strong>?</p>
                <p class="text-danger">Aksi ini tidak dapat dibatalkan!</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Batal</button>
                <form id="formHapus" method="POST" action="delete_pengguna.php">
                    <input type="hidden" name="id_pengguna" id="idPenggunaHapus">
                    <button type="submit" class="btn btn-danger">Hapus</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Include DataTables CSS & JS -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap4.min.css">
<script type="text/javascript" src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
<script type="text/javascript" src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap4.min.js"></script>

<script>
$(document).ready(function() {
    // Inisialisasi DataTables
    $('#dataTable').DataTable({
        "paging": false, // Nonaktifkan pagination bawaan DataTables
        "searching": false, // Nonaktifkan search bawaan DataTables
        "info": false, // Nonaktifkan info bawaan DataTables
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.11.5/i18n/Indonesian.json"
        }
    });

    // Modal konfirmasi hapus
    $('.btn-hapus').click(function() {
        var id = $(this).data('id');
        var nama = $(this).data('nama');
        
        $('#namaPenggunaHapus').text(nama);
        $('#idPenggunaHapus').val(id);
        $('#hapusModal').modal('show');
    });
});
</script>

<?php include 'footer.php'; ?>