<?php
session_start();
require_once 'config.php';

// 1. CEK LOGIN
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// 2. CEK HAK AKSES
// 0=Super Admin, 1=Admin, 2=Kasir, 3=Owner
$allowed_roles = ['0', '1', '2', '3']; 
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

// Inisialisasi keranjang belanja jika belum ada
if (!isset($_SESSION['keranjang'])) {
    $_SESSION['keranjang'] = [];
}

// Fungsi untuk format mata uang
function format_rupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

// Proses AJAX untuk berbagai aksi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    $response = ['status' => 'error', 'message' => 'Aksi tidak valid'];
    
    try {
        switch ($_POST['action']) {
            case 'tambah_produk':
                $id_barang = intval($_POST['id_barang']);
                $jumlah = intval($_POST['jumlah']);
                
                // Cek stok tersedia
                $query = "SELECT b.*, s.stok_akhir 
                          FROM barang b 
                          JOIN stok_toko s ON b.id_barang = s.id_barang 
                          WHERE b.id_barang = ? AND s.id_toko = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ii", $id_barang, $_SESSION['id_toko']);
                $stmt->execute();
                $result = $stmt->get_result();
                $produk = $result->fetch_assoc();
                
                if (!$produk) {
                    throw new Exception("Produk tidak ditemukan");
                }
                
                // Cek apakah stok mencukupi
                $stok_tersedia = $produk['stok_akhir'];
                $jumlah_di_keranjang = 0;
                
                foreach ($_SESSION['keranjang'] as $item) {
                    if ($item['id_barang'] == $id_barang) {
                        $jumlah_di_keranjang += $item['jumlah'];
                    }
                }
                
                if (($jumlah_di_keranjang + $jumlah) > $stok_tersedia) {
                    throw new Exception("Stok tidak mencukupi. Stok tersedia: " . $stok_tersedia);
                }
                
                // Cek apakah produk sudah ada di keranjang
                $produk_sudah_ada = false;
                foreach ($_SESSION['keranjang'] as &$item) {
                    if ($item['id_barang'] == $id_barang) {
                        $item['jumlah'] += $jumlah;
                        $produk_sudah_ada = true;
                        break;
                    }
                }
                
                // Jika produk belum ada di keranjang, tambahkan baru
                if (!$produk_sudah_ada) {
                    $_SESSION['keranjang'][] = [
                        'id_barang' => $id_barang,
                        'kode_barang' => $produk['kode_barang'],
                        'nama_barang' => $produk['nama_barang'],
                        'harga_jual' => $produk['harga_jual'],
                        'jumlah' => $jumlah,
                        'diskon_item' => 0,
                        'diskon_jenis' => 'nominal'
                    ];
                }
                
                $response = ['status' => 'success', 'message' => 'Produk berhasil ditambahkan'];
                break;
                
            case 'update_jumlah':
                $index = intval($_POST['index']);
                $jumlah = intval($_POST['jumlah']);
                
                if (!isset($_SESSION['keranjang'][$index])) {
                    throw new Exception("Item tidak ditemukan di keranjang");
                }
                
                // Cek stok tersedia
                $id_barang = $_SESSION['keranjang'][$index]['id_barang'];
                $query = "SELECT s.stok_akhir 
                          FROM stok_toko s 
                          WHERE s.id_barang = ? AND s.id_toko = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ii", $id_barang, $_SESSION['id_toko']);
                $stmt->execute();
                $result = $stmt->get_result();
                $stok = $result->fetch_assoc();
                
                if (!$stok) {
                    throw new Exception("Stok produk tidak ditemukan");
                }
                
                // Hitung total jumlah di keranjang (selain item yang sedang diupdate)
                $total_jumlah = 0;
                foreach ($_SESSION['keranjang'] as $i => $item) {
                    if ($i != $index && $item['id_barang'] == $id_barang) {
                        $total_jumlah += $item['jumlah'];
                    }
                }
                
                if (($total_jumlah + $jumlah) > $stok['stok_akhir']) {
                    throw new Exception("Stok tidak mencukupi. Stok tersedia: " . $stok['stok_akhir']);
                }
                
                $_SESSION['keranjang'][$index]['jumlah'] = $jumlah;
                $response = ['status' => 'success', 'message' => 'Jumlah berhasil diupdate'];
                break;
                
            case 'update_diskon':
                $index = intval($_POST['index']);
                $diskon = floatval($_POST['diskon']);
                $jenis_diskon = $_POST['jenis_diskon'];
                
                if (!isset($_SESSION['keranjang'][$index])) {
                    throw new Exception("Item tidak ditemukan di keranjang");
                }
                
                $_SESSION['keranjang'][$index]['diskon_item'] = $diskon;
                $_SESSION['keranjang'][$index]['diskon_jenis'] = $jenis_diskon;
                $response = ['status' => 'success', 'message' => 'Diskon berhasil diupdate'];
                break;
                
            case 'hapus_item':
                $index = intval($_POST['index']);
                
                if (!isset($_SESSION['keranjang'][$index])) {
                    throw new Exception("Item tidak ditemukan di keranjang");
                }
                
                array_splice($_SESSION['keranjang'], $index, 1);
                $response = ['status' => 'success', 'message' => 'Item berhasil dihapus'];
                break;
                
            case 'simpan_transaksi':
                // Validasi data
                if (empty($_SESSION['keranjang'])) {
                    throw new Exception("Keranjang belanja kosong");
                }
                
                $metode_bayar = intval($_POST['metode_bayar']);
                $jumlah_bayar = floatval(str_replace(['Rp', '.', ' '], '', $_POST['jumlah_bayar']));
                $id_pelanggan = !empty($_POST['id_pelanggan']) ? intval($_POST['id_pelanggan']) : NULL;
                $diskon_global = floatval(str_replace(['Rp', '.', ' '], '', $_POST['diskon_global']));
                $catatan = $conn->real_escape_string($_POST['catatan']);
                
                // Hitung total transaksi
                $subtotal = 0;
                foreach ($_SESSION['keranjang'] as $item) {
                    $harga = $item['harga_jual'];
                    $diskon = $item['diskon_item'];
                    
                    if ($item['diskon_jenis'] == 'persen') {
                        $diskon = $harga * ($diskon / 100);
                    }
                    
                    $subtotal += ($harga - $diskon) * $item['jumlah'];
                }
                
                $total_setelah_diskon = $subtotal - $diskon_global;
                $ppn = $total_setelah_diskon * 0.1; // PPN 10%
                $grand_total = $total_setelah_diskon + $ppn;
                
                // Validasi pembayaran
                if ($jumlah_bayar < $grand_total && $metode_bayar == 1) { // Cash harus cukup
                    throw new Exception("Jumlah pembayaran kurang dari total");
                }
                
                // Mulai transaksi database
                $conn->begin_transaction();
                
                try {
                    // Generate nomor transaksi
                    $prefix = 'PJL-' . date('Ymd') . '-';
                    $query = "SELECT MAX(no_transaksi) as last_number FROM penjualan WHERE no_transaksi LIKE '$prefix%'";
                    $result = $conn->query($query);
                    $last_number = $result->fetch_assoc()['last_number'];
                    
                    $next_number = 1;
                    if ($last_number) {
                        $last_num = intval(substr($last_number, strlen($prefix)));
                        $next_number = $last_num + 1;
                    }
                    
                    $no_transaksi = $prefix . str_pad($next_number, 4, '0', STR_PAD_LEFT);
                    
                    // Simpan transaksi ke tabel penjualan
                    $query = "INSERT INTO penjualan (
                        no_transaksi, id_toko, id_pelanggan, id_pengguna, id_metode, 
                        tanggal_penjualan, total_penjualan, diskon, ppn, grand_total, 
                        jumlah_bayar, kembalian, status_pembayaran, sisa_hutang, catatan
                    ) VALUES (?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    
                    $status_pembayaran = ($metode_bayar == 1 && $jumlah_bayar >= $grand_total) ? 'lunas' : 'hutang';
                    $sisa_hutang = ($status_pembayaran == 'hutang') ? ($grand_total - $jumlah_bayar) : 0;
                    $kembalian = ($metode_bayar == 1) ? ($jumlah_bayar - $grand_total) : 0;
                    
                    $stmt = $conn->prepare($query);
                    $stmt->bind_param(
                        "siiiiddddddsss", 
                        $no_transaksi, 
                        $_SESSION['id_toko'], 
                        $id_pelanggan, 
                        $_SESSION['id_pengguna'], 
                        $metode_bayar,
                        $subtotal,
                        $diskon_global,
                        $ppn,
                        $grand_total,
                        $jumlah_bayar,
                        $kembalian,
                        $status_pembayaran,
                        $sisa_hutang,
                        $catatan
                    );
                    
                    if (!$stmt->execute()) {
                        throw new Exception("Gagal menyimpan transaksi: " . $stmt->error);
                    }
                    
                    $id_penjualan = $conn->insert_id;
                    
                    // Simpan detail penjualan
                    foreach ($_SESSION['keranjang'] as $item) {
                        $diskon = $item['diskon_item'];
                        $jenis_diskon = $item['diskon_jenis'];
                        $subtotal_item = ($item['harga_jual'] - $diskon) * $item['jumlah'];
                        
                        $query = "INSERT INTO detail_penjualan (
                            id_penjualan, id_barang, jumlah, harga_jual, diskon_item, 
                            diskon_jenis, subtotal
                        ) VALUES (?, ?, ?, ?, ?, ?, ?)";
                        
                        $stmt = $conn->prepare($query);
                        $stmt->bind_param(
                            "iiiddsd", 
                            $id_penjualan, 
                            $item['id_barang'], 
                            $item['jumlah'], 
                            $item['harga_jual'], 
                            $diskon,
                            $jenis_diskon,
                            $subtotal_item
                        );
                        
                        if (!$stmt->execute()) {
                            throw new Exception("Gagal menyimpan detail penjualan: " . $stmt->error);
                        }
                        
                        // Kurangi stok
                        $query = "UPDATE stok_toko 
                                 SET stok_keluar = stok_keluar + ?, 
                                     stok_akhir = stok_awal + stok_masuk - (stok_keluar + ?)
                                 WHERE id_barang = ? AND id_toko = ?";
                        
                        $stmt = $conn->prepare($query);
                        $stmt->bind_param(
                            "iiii", 
                            $item['jumlah'], 
                            $item['jumlah'], 
                            $item['id_barang'], 
                            $_SESSION['id_toko']
                        );
                        
                        if (!$stmt->execute()) {
                            throw new Exception("Gagal mengupdate stok: " . $stmt->error);
                        }
                    }
                    
                    // Jika ada hutang, simpan ke tabel hutang_pelanggan
                    if ($status_pembayaran == 'hutang' && $id_pelanggan) {
                        $tanggal_jatuh_tempo = date('Y-m-d', strtotime('+30 days'));
                        
                        $query = "INSERT INTO hutang_pelanggan (
                            id_penjualan, id_pelanggan, tanggal_hutang, tanggal_jatuh_tempo,
                            jumlah_hutang, jumlah_bayar, sisa_hutang, status, keterangan
                        ) VALUES (?, ?, CURDATE(), ?, ?, ?, ?, ?, ?)";
                        
                        $keterangan = "Hutang pembelian " . $no_transaksi;
                        $status = ($sisa_hutang > 0) ? 'belum_lunas' : 'lunas';
                        
                        $stmt = $conn->prepare($query);
                        $stmt->bind_param(
                            "iisdddss", 
                            $id_penjualan, 
                            $id_pelanggan,
                            $tanggal_jatuh_tempo,
                            $grand_total,
                            $jumlah_bayar,
                            $sisa_hutang,
                            $status,
                            $keterangan
                        );
                        
                        if (!$stmt->execute()) {
                            throw new Exception("Gagal menyimpan data hutang: " . $stmt->error);
                        }
                    }
                    
                    // Commit transaksi
                    $conn->commit();
                    
                    // Kosongkan keranjang
                    $_SESSION['keranjang'] = [];
                    
                    // Set response
                    $response = [
                        'status' => 'success', 
                        'message' => 'Transaksi berhasil disimpan',
                        'no_transaksi' => $no_transaksi,
                        'id_penjualan' => $id_penjualan
                    ];
                    
                } catch (Exception $e) {
                    $conn->rollback();
                    throw $e;
                }
                break;
                
            case 'cari_produk':
                $keyword = $conn->real_escape_string($_POST['keyword']);
                
                $query = "SELECT b.id_barang, b.kode_barang, b.nama_barang, b.harga_jual, 
                         s.stok_akhir, k.nama_kategori
                         FROM barang b
                         JOIN stok_toko s ON b.id_barang = s.id_barang
                         JOIN kategori k ON b.id_kategori = k.id_kategori
                         WHERE (b.kode_barang LIKE '%$keyword%' OR 
                                b.nama_barang LIKE '%$keyword%' OR 
                                b.barcode LIKE '%$keyword%') AND
                               s.id_toko = " . $_SESSION['id_toko'] . "
                         LIMIT 10";
                
                $result = $conn->query($query);
                $produk = [];
                
                while ($row = $result->fetch_assoc()) {
                    $produk[] = $row;
                }
                
                $response = ['status' => 'success', 'data' => $produk];
                break;
                
            case 'cari_pelanggan':
                $keyword = $conn->real_escape_string($_POST['keyword']);
                
                $query = "SELECT * FROM pelanggan 
                         WHERE nama_pelanggan LIKE '%$keyword%' OR 
                               telepon LIKE '%$keyword%' OR
                               kode_pelanggan LIKE '%$keyword%'
                         LIMIT 10";
                
                $result = $conn->query($query);
                $pelanggan = [];
                
                while ($row = $result->fetch_assoc()) {
                    $pelanggan[] = $row;
                }
                
                $response = ['status' => 'success', 'data' => $pelanggan];
                break;
        }
    } catch (Exception $e) {
        $response = ['status' => 'error', 'message' => $e->getMessage()];
    }
    
    echo json_encode($response);
    exit();
}

// Hitung total transaksi
$subtotal = 0;
$total_diskon_item = 0;
$total_setelah_diskon = 0;
$ppn = 0;
$grand_total = 0;

foreach ($_SESSION['keranjang'] as $item) {
    $harga = $item['harga_jual'];
    $diskon = $item['diskon_item'];
    
    if ($item['diskon_jenis'] == 'persen') {
        $diskon = $harga * ($diskon / 100);
    }
    
    $subtotal += $harga * $item['jumlah'];
    $total_diskon_item += $diskon * $item['jumlah'];
}

$total_setelah_diskon = $subtotal - $total_diskon_item;
$ppn = $total_setelah_diskon * 0.1; // PPN 10%
$grand_total = $total_setelah_diskon + $ppn;

include 'header.php';
?>

<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Kasir</h1>
        <div class="d-none d-sm-inline-block">
            <span class="text-muted">Tanggal: <?php echo date('d M Y'); ?></span>
        </div>
    </div>
    
    <div class="row">
        <!-- Kolom Pencarian Produk -->
        <div class="col-lg-5 mb-4">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Tambah Produk</h6>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label for="cariProduk">Cari Produk (Kode/Nama)</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="cariProduk" placeholder="Scan barcode atau ketik nama produk..." autofocus>
                            <button class="btn btn-primary" id="btnCariProduk">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div id="hasilPencarianProduk" class="mt-3" style="display: none;">
                        <div class="list-group" id="listProduk"></div>
                    </div>
                    
                    <div id="detailProduk" class="mt-3" style="display: none;">
                        <div class="card">
                            <div class="card-body">
                                <h5 id="namaProduk" class="card-title"></h5>
                                <p class="card-text">
                                    <span id="kodeProduk" class="badge bg-primary"></span>
                                    <span id="kategoriProduk" class="badge bg-secondary ms-1"></span>
                                    <span id="stokProduk" class="badge bg-success ms-1"></span>
                                </p>
                                <p class="card-text">
                                    <strong>Harga:</strong> <span id="hargaProduk"></span>
                                </p>
                                <div class="form-group">
                                    <label for="jumlahProduk">Jumlah</label>
                                    <input type="number" min="1" value="1" class="form-control" id="jumlahProduk">
                                </div>
                                <button class="btn btn-primary mt-2" id="btnTambahKeKeranjang">
                                    <i class="fas fa-cart-plus"></i> Tambah ke Keranjang
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Daftar Keranjang Belanja -->
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex justify-content-between align-items-center">
                    <h6 class="m-0 font-weight-bold text-primary">Keranjang Belanja</h6>
                    <span class="badge bg-primary"><?php echo count($_SESSION['keranjang']); ?> item</span>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered" id="keranjangTable">
                            <thead>
                                <tr>
                                    <th width="5%">No</th>
                                    <th width="35%">Produk</th>
                                    <th width="15%">Harga</th>
                                    <th width="15%">Jumlah</th>
                                    <th width="15%">Diskon</th>
                                    <th width="15%">Subtotal</th>
                                    <th width="5%">Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($_SESSION['keranjang'])): ?>
                                    <tr>
                                        <td colspan="7" class="text-center">Keranjang belanja kosong</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($_SESSION['keranjang'] as $index => $item): ?>
                                        <?php
                                        $diskon = $item['diskon_item'];
                                        if ($item['diskon_jenis'] == 'persen') {
                                            $diskon = $item['harga_jual'] * ($diskon / 100);
                                        }
                                        $subtotal_item = ($item['harga_jual'] - $diskon) * $item['jumlah'];
                                        ?>
                                        <tr data-id="<?php echo $item['id_barang']; ?>">
                                            <td><?php echo $index + 1; ?></td>
                                            <td><?php echo $item['nama_barang']; ?></td>
                                            <td class="text-right"><?php echo format_rupiah($item['harga_jual']); ?></td>
                                            <td>
                                                <input type="number" min="1" value="<?php echo $item['jumlah']; ?>" 
                                                       class="form-control form-control-sm update-jumlah" 
                                                       data-index="<?php echo $index; ?>">
                                            </td>
                                            <td>
                                                <div class="input-group input-group-sm">
                                                    <input type="number" min="0" 
                                                           value="<?php echo $item['diskon_item']; ?>" 
                                                           class="form-control update-diskon" 
                                                           data-index="<?php echo $index; ?>">
                                                    <select class="form-select form-select-sm diskon-jenis" 
                                                            data-index="<?php echo $index; ?>">
                                                        <option value="nominal" <?php echo ($item['diskon_jenis'] == 'nominal') ? 'selected' : ''; ?>>Rp</option>
                                                        <option value="persen" <?php echo ($item['diskon_jenis'] == 'persen') ? 'selected' : ''; ?>>%</option>
                                                    </select>
                                                </div>
                                            </td>
                                            <td class="text-right"><?php echo format_rupiah($subtotal_item); ?></td>
                                            <td class="text-center">
                                                <button class="btn btn-sm btn-danger hapus-item" 
                                                        data-index="<?php echo $index; ?>">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Kolom Pembayaran -->
        <div class="col-lg-7 mb-4">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Pembayaran</h6>
                </div>
                <div class="card-body">
                    <form id="formPembayaran">
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="cariPelanggan">Pelanggan (Opsional)</label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="cariPelanggan" placeholder="Cari pelanggan...">
                                        <button class="btn btn-outline-secondary" type="button" id="btnCariPelanggan">
                                            <i class="fas fa-search"></i>
                                        </button>
                                    </div>
                                    <input type="hidden" id="idPelanggan" name="id_pelanggan">
                                    <div id="infoPelanggan" class="mt-2 small" style="display: none;">
                                        <span id="namaPelanggan" class="fw-bold"></span> - 
                                        <span id="teleponPelanggan"></span>
                                        <button type="button" class="btn btn-sm btn-link p-0 ms-2" id="hapusPelanggan">
                                            <i class="fas fa-times text-danger"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="metodeBayar">Metode Pembayaran</label>
                                    <select class="form-control" id="metodeBayar" name="metode_bayar" required>
                                        <option value="1">Tunai</option>
                                        <option value="2">Transfer Bank</option>
                                        <option value="3">QRIS</option>
                                        <option value="4">E-Wallet</option>
                                        <option value="5">Kartu Debit</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="diskonGlobal">Promo & Diskon</label>
                                    <div class="input-group">
                                        <span class="input-group-text">Rp</span>
                                        <input type="text" class="form-control" id="diskonGlobal" name="diskon_global" value="0">
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="catatan">Catatan (Opsional)</label>
                                    <textarea class="form-control" id="catatan" name="catatan" rows="1"></textarea>
                                </div>
                            </div>
                        </div>
                        
                        <div class="table-responsive mb-4">
                            <table class="table table-bordered">
                                <tbody>
                                    <tr>
                                        <th width="70%">Subtotal</th>
                                        <td width="30%" class="text-right" id="subtotalDisplay"><?php echo format_rupiah($subtotal); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Diskon Item</th>
                                        <td class="text-right" id="diskonItemDisplay"><?php echo format_rupiah($total_diskon_item); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Total Setelah Diskon</th>
                                        <td class="text-right" id="totalSetelahDiskonDisplay"><?php echo format_rupiah($total_setelah_diskon); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Diskon Global</th>
                                        <td class="text-right" id="diskonGlobalDisplay">Rp 0</td>
                                    </tr>
                                    <tr>
                                        <th>PPN (10%)</th>
                                        <td class="text-right" id="ppnDisplay"><?php echo format_rupiah($ppn); ?></td>
                                    </tr>
                                    <tr class="table-active">
                                        <th>Grand Total</th>
                                        <td class="text-right fw-bold" id="grandTotalDisplay"><?php echo format_rupiah($grand_total); ?></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="jumlahBayar">Jumlah Bayar</label>
                                    <div class="input-group">
                                        <span class="input-group-text">Rp</span>
                                        <input type="text" class="form-control" id="jumlahBayar" name="jumlah_bayar" required>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="kembalian">Kembalian</label>
                                    <div class="input-group">
                                        <span class="input-group-text">Rp</span>
                                        <input type="text" class="form-control" id="kembalian" readonly>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-success btn-lg py-3" id="btnSimpanTransaksi">
                                <i class="fas fa-save"></i> Simpan Transaksi
                            </button>
                            <button type="button" class="btn btn-danger btn-lg py-3" id="btnBatalTransaksi">
                                <i class="fas fa-times"></i> Batalkan Transaksi
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Konfirmasi -->
<div class="modal fade" id="confirmModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="confirmModalTitle">Konfirmasi</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="confirmModalBody">
                Apakah Anda yakin ingin menyimpan transaksi ini?
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" id="confirmSave">Ya, Simpan</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal Sukses -->
<div class="modal fade" id="successModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title">Transaksi Berhasil</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="successModalBody">
                Transaksi berhasil disimpan.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                <a href="#" class="btn btn-primary" id="btnCetakStruk">Cetak Struk</a>
            </div>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Format input uang
    $('input[id^="diskon"], input[id="jumlahBayar"]').on('keyup', function() {
        let value = $(this).val().replace(/[^0-9]/g, '');
        if (value === '') value = '0';
        $(this).val(formatRupiah(value));
    });
    
    // Fungsi format rupiah
    function formatRupiah(angka) {
        return 'Rp ' + angka.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ".");
    }
    
    // Fungsi parse rupiah ke angka
    function parseRupiah(rupiah) {
        return parseInt(rupiah.replace(/[^0-9]/g, ''));
    }
    
    // Pencarian produk
    $('#cariProduk').on('keyup', function() {
        let keyword = $(this).val();
        if (keyword.length >= 2) {
            $.post('kasir.php', {
                action: 'cari_produk',
                keyword: keyword
            }, function(response) {
                if (response.status === 'success') {
                    if (response.data.length > 0) {
                        let html = '';
                        response.data.forEach(function(produk) {
                            html += `<a href="#" class="list-group-item list-group-item-action pilih-produk" 
                                     data-id="${produk.id_barang}" 
                                     data-kode="${produk.kode_barang}" 
                                     data-nama="${produk.nama_barang}" 
                                     data-harga="${produk.harga_jual}" 
                                     data-stok="${produk.stok_akhir}" 
                                     data-kategori="${produk.nama_kategori}">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1">${produk.nama_barang}</h6>
                                    <small>${formatRupiah(produk.harga_jual)}</small>
                                </div>
                                <small class="text-muted">${produk.kode_barang} | Stok: ${produk.stok_akhir}</small>
                            </a>`;
                        });
                        $('#listProduk').html(html);
                        $('#hasilPencarianProduk').show();
                        $('#detailProduk').hide();
                    } else {
                        $('#listProduk').html('<div class="list-group-item">Produk tidak ditemukan</div>');
                        $('#hasilPencarianProduk').show();
                        $('#detailProduk').hide();
                    }
                }
            }, 'json');
        } else {
            $('#hasilPencarianProduk').hide();
        }
    });
    
    // Pilih produk dari hasil pencarian
    $(document).on('click', '.pilih-produk', function(e) {
        e.preventDefault();
        
        $('#namaProduk').text($(this).data('nama'));
        $('#kodeProduk').text($(this).data('kode'));
        $('#kategoriProduk').text($(this).data('kategori'));
        $('#hargaProduk').text(formatRupiah($(this).data('harga')));
        $('#stokProduk').text('Stok: ' + $(this).data('stok'));
        $('#jumlahProduk').attr('max', $(this).data('stok'));
        $('#jumlahProduk').val(1);
        
        $('#hasilPencarianProduk').hide();
        $('#detailProduk').show();
        $('#jumlahProduk').focus();
        
        // Simpan data produk yang dipilih
        $('#detailProduk').data('produk', $(this).data());
    });
    
    // Tambah produk ke keranjang
    $('#btnTambahKeKeranjang').click(function() {
        let produk = $('#detailProduk').data('produk');
        let jumlah = parseInt($('#jumlahProduk').val());
        
        if (jumlah < 1) {
            alert('Jumlah tidak valid');
            return;
        }
        
        $.post('kasir.php', {
            action: 'tambah_produk',
            id_barang: produk.id,
            jumlah: jumlah
        }, function(response) {
            if (response.status === 'success') {
                location.reload(); // Refresh untuk update keranjang
            } else {
                alert(response.message);
            }
        }, 'json');
    });
    
    // Update jumlah item di keranjang
    $(document).on('change', '.update-jumlah', function() {
        let index = $(this).data('index');
        let jumlah = parseInt($(this).val());
                if (jumlah < 1) {
            alert('Jumlah tidak valid');
            $(this).val(1);
            return;
        }

        $.post('kasir.php', {
            action: 'update_jumlah',
            index: index,
            jumlah: jumlah
        }, function(response) {
            if (response.status === 'success') {
                location.reload(); // Refresh untuk update keranjang
            } else {
                alert(response.message);
                location.reload(); // Refresh untuk mendapatkan data terbaru
            }
        }, 'json');
    });

    // Update diskon item di keranjang
    $(document).on('change', '.update-diskon, .diskon-jenis', function() {
        let index = $(this).data('index');
        let diskon = $('.update-diskon[data-index="' + index + '"]').val();
        let jenisDiskon = $('.diskon-jenis[data-index="' + index + '"]').val();

        $.post('kasir.php', {
            action: 'update_diskon',
            index: index,
            diskon: diskon,
            jenis_diskon: jenisDiskon
        }, function(response) {
            if (response.status === 'success') {
                location.reload(); // Refresh untuk update keranjang
            } else {
                alert(response.message);
                location.reload(); // Refresh untuk mendapatkan data terbaru
            }
        }, 'json');
    });

    // Hapus item dari keranjang
    $(document).on('click', '.hapus-item', function() {
        if (confirm('Apakah Anda yakin ingin menghapus item ini dari keranjang?')) {
            let index = $(this).data('index');

            $.post('kasir.php', {
                action: 'hapus_item',
                index: index
            }, function(response) {
                if (response.status === 'success') {
                    location.reload(); // Refresh untuk update keranjang
                } else {
                    alert(response.message);
                }
            }, 'json');
        }
    });

    // Pencarian pelanggan
    $('#cariPelanggan').on('keyup', function() {
        let keyword = $(this).val();
        if (keyword.length >= 2) {
            $.post('kasir.php', {
                action: 'cari_pelanggan',
                keyword: keyword
            }, function(response) {
                if (response.status === 'success') {
                    if (response.data.length > 0) {
                        let html = '<div class="list-group">';
                        response.data.forEach(function(pelanggan) {
                            html += `<a href="#" class="list-group-item list-group-item-action pilih-pelanggan" 
                                     data-id="${pelanggan.id_pelanggan}" 
                                     data-nama="${pelanggan.nama_pelanggan}" 
                                     data-telepon="${pelanggan.telepon}">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1">${pelanggan.nama_pelanggan}</h6>
                                    <small>${pelanggan.kode_pelanggan || ''}</small>
                                </div>
                                <small class="text-muted">${pelanggan.telepon || 'No. Telp tidak ada'}</small>
                            </a>`;
                        });
                        html += '</div>';
                        
                        // Tampilkan hasil pencarian
                        $('#cariPelanggan').autocomplete({
                            source: response.data.map(p => p.nama_pelanggan),
                            select: function(event, ui) {
                                let selected = response.data.find(p => p.nama_pelanggan === ui.item.value);
                                if (selected) {
                                    pilihPelanggan(selected);
                                }
                            }
                        });
                    }
                }
            }, 'json');
        }
    });

    // Pilih pelanggan dari hasil pencarian
    function pilihPelanggan(pelanggan) {
        $('#idPelanggan').val(pelanggan.id_pelanggan);
        $('#namaPelanggan').text(pelanggan.nama_pelanggan);
        $('#teleponPelanggan').text(pelanggan.telepon || 'No. Telp tidak ada');
        $('#infoPelanggan').show();
        $('#cariPelanggan').val('');
    }

    $(document).on('click', '.pilih-pelanggan', function(e) {
        e.preventDefault();
        pilihPelanggan($(this).data());
    });

    // Hapus pelanggan yang dipilih
    $('#hapusPelanggan').click(function() {
        $('#idPelanggan').val('');
        $('#infoPelanggan').hide();
    });

    // Hitung kembalian
    $('#jumlahBayar').on('keyup', function() {
        hitungKembalian();
    });

    $('#metodeBayar').change(function() {
        hitungKembalian();
    });

    function hitungKembalian() {
        let grandTotal = parseRupiah($('#grandTotalDisplay').text());
        let jumlahBayar = parseRupiah($('#jumlahBayar').val());
        let metodeBayar = $('#metodeBayar').val();

        if (metodeBayar == 1) { // Tunai
            if (jumlahBayar >= grandTotal) {
                $('#kembalian').val(formatRupiah(jumlahBayar - grandTotal));
            } else {
                $('#kembalian').val(formatRupiah(0));
            }
        } else {
            $('#kembalian').val(formatRupiah(0));
        }
    }

    // Update diskon global
    $('#diskonGlobal').on('keyup', function() {
        let diskon = parseRupiah($(this).val());
        $('#diskonGlobalDisplay').text(formatRupiah(diskon));
        
        // Hitung ulang total setelah diskon global
        let totalSetelahDiskon = parseRupiah($('#totalSetelahDiskonDisplay').text());
        let diskonGlobal = parseRupiah($('#diskonGlobal').val());
        let ppn = (totalSetelahDiskon - diskonGlobal) * 0.1;
        let grandTotal = (totalSetelahDiskon - diskonGlobal) + ppn;
        
        $('#ppnDisplay').text(formatRupiah(ppn));
        $('#grandTotalDisplay').text(formatRupiah(grandTotal));
    });

    // Simpan transaksi
    $('#formPembayaran').submit(function(e) {
        e.preventDefault();
        
        // Validasi keranjang tidak kosong
        if ($('#keranjangTable tbody tr:not(.empty-row)').length == 0) {
            alert('Keranjang belanja kosong');
            return;
        }
        
        // Validasi jumlah bayar untuk metode tunai
        let metodeBayar = $('#metodeBayar').val();
        let jumlahBayar = parseRupiah($('#jumlahBayar').val());
        let grandTotal = parseRupiah($('#grandTotalDisplay').text());
        
        if (metodeBayar == 1 && jumlahBayar < grandTotal) {
            alert('Jumlah pembayaran kurang dari total');
            return;
        }
        
        // Tampilkan modal konfirmasi
        $('#confirmModal').modal('show');
    });

    // Konfirmasi simpan transaksi
    $('#confirmSave').click(function() {
        $.post('kasir.php', {
            action: 'simpan_transaksi',
            metode_bayar: $('#metodeBayar').val(),
            jumlah_bayar: parseRupiah($('#jumlahBayar').val()),
            id_pelanggan: $('#idPelanggan').val(),
            diskon_global: parseRupiah($('#diskonGlobal').val()),
            catatan: $('#catatan').val()
        }, function(response) {
            $('#confirmModal').modal('hide');
            
            if (response.status === 'success') {
                // Tampilkan modal sukses
                $('#successModalBody').html(`
                    <p>Transaksi <strong>${response.no_transaksi}</strong> berhasil disimpan.</p>
                    <p>Total: <strong>${$('#grandTotalDisplay').text()}</strong></p>
                `);
                
                // Set link cetak struk
                $('#btnCetakStruk').attr('href', 'cetak_struk.php?id=' + response.id_penjualan);
                
                $('#successModal').modal('show');
            } else {
                alert(response.message);
            }
        }, 'json');
    });

    // Batalkan transaksi
    $('#btnBatalTransaksi').click(function() {
        if (confirm('Apakah Anda yakin ingin membatalkan transaksi ini? Semua item di keranjang akan dihapus.')) {
            $.post('kasir.php', {
                action: 'batal_transaksi'
            }, function(response) {
                if (response.status === 'success') {
                    location.reload();
                } else {
                    alert(response.message);
                }
            }, 'json');
        }
    });

    // Ketika modal sukses ditutup
    $('#successModal').on('hidden.bs.modal', function() {
        location.reload(); // Refresh halaman untuk transaksi baru
    });

    // Focus ke input pencarian produk saat halaman dimuat
    $('#cariProduk').focus();
});
</script>

<?php include 'footer.php'; ?>