<?php
// Start session dan validasi login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Validasi hak akses (hanya Super Admin, Admin dan Owner)
$allowed_roles = ['0', '1', '3']; // Sesuaikan dengan id_role super admin, admin dan owner
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

include 'header.php';
require_once 'config.php';

// Fungsi untuk format mata uang
function format_rupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

// Set default tanggal (awal bulan - hari ini)
$tanggal_dari = date('Y-m-01');
$tanggal_sampai = date('Y-m-d');

// Ambil parameter filter jika ada
if (isset($_GET['tanggal_dari']) && !empty($_GET['tanggal_dari'])) {
    $tanggal_dari = date('Y-m-d', strtotime($_GET['tanggal_dari']));
}

if (isset($_GET['tanggal_sampai']) && !empty($_GET['tanggal_sampai'])) {
    $tanggal_sampai = date('Y-m-d', strtotime($_GET['tanggal_sampai']));
}

$id_toko = isset($_GET['id_toko']) ? (int)$_GET['id_toko'] : 0;
$id_kategori = isset($_GET['id_kategori']) ? (int)$_GET['id_kategori'] : 0;

// Query untuk data laporan laba
$query = "SELECT 
            p.id_penjualan,
            p.no_transaksi,
            DATE(p.tanggal_penjualan) as tanggal,
            t.nama_toko,
            SUM(dp.jumlah) as jumlah_item,
            SUM(dp.subtotal) as total_penjualan,
            SUM(dp.jumlah * b.harga_beli) as total_hpp,
            SUM(dp.subtotal - (dp.jumlah * b.harga_beli)) as laba_kotor,
            ROUND((SUM(dp.subtotal - (dp.jumlah * b.harga_beli)) / SUM(dp.subtotal)) * 100, 2) as margin
          FROM penjualan p
          JOIN detail_penjualan dp ON p.id_penjualan = dp.id_penjualan
          JOIN barang b ON dp.id_barang = b.id_barang
          JOIN toko t ON p.id_toko = t.id_toko
          WHERE DATE(p.tanggal_penjualan) BETWEEN ? AND ?
          AND p.status_pembayaran != 'dibatalkan'";

$params = [$tanggal_dari, $tanggal_sampai];
$types = "ss";

// Filter berdasarkan toko
if ($id_toko > 0) {
    $query .= " AND p.id_toko = ?";
    $params[] = $id_toko;
    $types .= "i";
} elseif ($_SESSION['id_role'] != 0) { // Bukan Super Admin
    $query .= " AND p.id_toko = ?";
    $params[] = $_SESSION['id_toko'];
    $types .= "i";
}

// Filter berdasarkan kategori
if ($id_kategori > 0) {
    $query .= " AND b.id_kategori = ?";
    $params[] = $id_kategori;
    $types .= "i";
}

$query .= " GROUP BY p.id_penjualan
            ORDER BY p.tanggal_penjualan DESC";

// Prepare statement
$stmt = mysqli_prepare($conn, $query);
if ($stmt) {
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
} else {
    die("Error in preparing statement: " . mysqli_error($conn));
}

// Hitung total
$total_penjualan = 0;
$total_hpp = 0;
$total_laba = 0;
$data_laba = [];
while ($row = mysqli_fetch_assoc($result)) {
    $data_laba[] = $row;
    $total_penjualan += $row['total_penjualan'];
    $total_hpp += $row['total_hpp'];
    $total_laba += $row['laba_kotor'];
}

// Query untuk data laba per kategori
$query_kategori = "SELECT 
                    k.id_kategori,
                    k.nama_kategori,
                    SUM(dp.subtotal) as total_penjualan,
                    SUM(dp.jumlah * b.harga_beli) as total_hpp,
                    SUM(dp.subtotal - (dp.jumlah * b.harga_beli)) as laba_kotor,
                    ROUND((SUM(dp.subtotal - (dp.jumlah * b.harga_beli)) / SUM(dp.subtotal)) * 100, 2) as margin
                  FROM penjualan p
                  JOIN detail_penjualan dp ON p.id_penjualan = dp.id_penjualan
                  JOIN barang b ON dp.id_barang = b.id_barang
                  JOIN kategori k ON b.id_kategori = k.id_kategori
                  WHERE DATE(p.tanggal_penjualan) BETWEEN ? AND ?
                  AND p.status_pembayaran != 'dibatalkan'";

if ($id_toko > 0 || $_SESSION['id_role'] != 0) {
    $query_kategori .= " AND p.id_toko = ?";
    $params_kategori = [$tanggal_dari, $tanggal_sampai, ($id_toko > 0 ? $id_toko : $_SESSION['id_toko'])];
    $types_kategori = "ssi";
} else {
    $params_kategori = [$tanggal_dari, $tanggal_sampai];
    $types_kategori = "ss";
}

$query_kategori .= " GROUP BY k.id_kategori
                     ORDER BY laba_kotor DESC";

$stmt_kategori = mysqli_prepare($conn, $query_kategori);
if ($stmt_kategori) {
    mysqli_stmt_bind_param($stmt_kategori, $types_kategori, ...$params_kategori);
    mysqli_stmt_execute($stmt_kategori);
    $result_kategori = mysqli_stmt_get_result($stmt_kategori);
}

// Query untuk dropdown filter
$query_toko = "SELECT * FROM toko WHERE status_aktif = 1";
if ($_SESSION['id_role'] != 0) { // Bukan Super Admin
    $query_toko .= " AND id_toko = " . $_SESSION['id_toko'];
}
$result_toko = mysqli_query($conn, $query_toko);

$query_kategori_filter = "SELECT * FROM kategori ORDER BY nama_kategori";
$result_kategori_filter = mysqli_query($conn, $query_kategori_filter);
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Laporan Laba Rugi</h1>
    </div>

    <!-- Filter Form -->
    <div class="card mb-4 animate-fade-in">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary-custom">Filter Laporan</h6>
        </div>
        <div class="card-body">
            <form method="GET" action="laporan_laba.php">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="tanggal_dari">Tanggal Dari</label>
                            <input type="date" class="form-control" id="tanggal_dari" name="tanggal_dari" 
                                   value="<?php echo htmlspecialchars($tanggal_dari); ?>">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="tanggal_sampai">Tanggal Sampai</label>
                            <input type="date" class="form-control" id="tanggal_sampai" name="tanggal_sampai" 
                                   value="<?php echo htmlspecialchars($tanggal_sampai); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="id_toko">Toko</label>
                            <select class="form-control" id="id_toko" name="id_toko" <?php echo ($_SESSION['id_role'] != 0) ? 'disabled' : ''; ?>>
                                <option value="0">Semua Toko</option>
                                <?php while ($toko = mysqli_fetch_assoc($result_toko)): ?>
                                    <option value="<?php echo $toko['id_toko']; ?>" 
                                        <?php echo ($id_toko == $toko['id_toko']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($toko['nama_toko']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <?php if ($_SESSION['id_role'] != 0): ?>
                                <input type="hidden" name="id_toko" value="<?php echo $_SESSION['id_toko']; ?>">
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="id_kategori">Kategori</label>
                            <select class="form-control" id="id_kategori" name="id_kategori">
                                <option value="0">Semua Kategori</option>
                                <?php while ($kategori = mysqli_fetch_assoc($result_kategori_filter)): ?>
                                    <option value="<?php echo $kategori['id_kategori']; ?>" 
                                        <?php echo ($id_kategori == $kategori['id_kategori']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($kategori['nama_kategori']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary-custom mr-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="laporan_laba.php" class="btn btn-secondary mr-2">
                            <i class="fas fa-sync-alt"></i> Reset
                        </a>
                        <button type="button" class="btn btn-success" id="btnExportPDF">
                            <i class="fas fa-file-pdf"></i> Export PDF
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Ringkasan Laba -->
    <div class="row mb-4">
        <!-- Total Penjualan -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-success h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Total Penjualan</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo format_rupiah($total_penjualan); ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-shopping-cart fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Total HPP -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-danger h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                Total HPP</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo format_rupiah($total_hpp); ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-boxes fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Laba Kotor -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-primary h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col me-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Laba Kotor</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo format_rupiah($total_laba); ?></div>
                            <div class="mt-1 text-muted small">
                                Margin: <?php echo round(($total_laba / ($total_penjualan ?: 1)) * 100, 2); ?>%
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-chart-line fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Data Table -->
    <div class="card mb-4 animate-fade-in">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary-custom">Detail Transaksi</h6>
            <div class="text-muted">
                <span class="badge bg-light text-dark">
                    <?php echo count($data_laba); ?> Transaksi
                </span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-hover" id="dataTable" width="100%" cellspacing="0">
                    <thead class="thead-light">
                        <tr>
                            <th>No. Transaksi</th>
                            <th>Tanggal</th>
                            <th>Toko</th>
                            <th>Item</th>
                            <th>Penjualan</th>
                            <th>HPP</th>
                            <th>Laba Kotor</th>
                            <th>Margin</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($data_laba as $laba): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($laba['no_transaksi']); ?></td>
                            <td><?php echo date('d M Y', strtotime($laba['tanggal'])); ?></td>
                            <td><?php echo htmlspecialchars($laba['nama_toko']); ?></td>
                            <td class="text-center"><?php echo $laba['jumlah_item']; ?></td>
                            <td class="text-right"><?php echo format_rupiah($laba['total_penjualan']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($laba['total_hpp']); ?></td>
                            <td class="text-right font-weight-bold <?php echo ($laba['laba_kotor'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo format_rupiah($laba['laba_kotor']); ?>
                            </td>
                            <td class="text-right <?php echo ($laba['margin'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo $laba['margin']; ?>%
                            </td>
                            <td>
                                <a href="detail_penjualan.php?id=<?php echo $laba['id_penjualan']; ?>" 
                                   class="btn btn-sm btn-primary-custom" title="Detail">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr class="font-weight-bold">
                            <td colspan="4" class="text-right">TOTAL</td>
                            <td class="text-right"><?php echo format_rupiah($total_penjualan); ?></td>
                            <td class="text-right"><?php echo format_rupiah($total_hpp); ?></td>
                            <td class="text-right <?php echo ($total_laba >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo format_rupiah($total_laba); ?>
                            </td>
                            <td class="text-right <?php echo (($total_laba / ($total_penjualan ?: 1)) * 100 >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo round(($total_laba / ($total_penjualan ?: 1)) * 100, 2); ?>%
                            </td>
                            <td></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>

    <!-- Laba per Kategori -->
    <div class="card animate-fade-in">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary-custom">Laba per Kategori</h6>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-hover" id="dataTableKategori" width="100%" cellspacing="0">
                    <thead class="thead-light">
                        <tr>
                            <th>Kategori</th>
                            <th>Penjualan</th>
                            <th>HPP</th>
                            <th>Laba Kotor</th>
                            <th>Margin</th>
                            <th>% Kontribusi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (isset($result_kategori)): 
                            while ($kategori = mysqli_fetch_assoc($result_kategori)): 
                                $kontribusi = ($kategori['laba_kotor'] / ($total_laba ?: 1)) * 100;
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($kategori['nama_kategori']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($kategori['total_penjualan']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($kategori['total_hpp']); ?></td>
                            <td class="text-right font-weight-bold <?php echo ($kategori['laba_kotor'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo format_rupiah($kategori['laba_kotor']); ?>
                            </td>
                            <td class="text-right <?php echo ($kategori['margin'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                <?php echo $kategori['margin']; ?>%
                            </td>
                            <td>
                                <div class="progress">
                                    <div class="progress-bar <?php echo ($kategori['laba_kotor'] >= 0) ? 'bg-success' : 'bg-danger'; ?>" 
                                         role="progressbar" style="width: <?php echo abs($kontribusi); ?>%" 
                                         aria-valuenow="<?php echo abs($kontribusi); ?>" 
                                         aria-valuemin="0" aria-valuemax="100">
                                        <?php echo round(abs($kontribusi), 2); ?>%
                                    </div>
                                </div>
                            </td>
                        </tr>
                        <?php endwhile; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</main>

<!-- Script untuk DataTables -->
<script>
$(document).ready(function() {
    // Inisialisasi DataTables untuk transaksi
    $('#dataTable').DataTable({
        dom: 'Bfrtip',
        buttons: [
            {
                extend: 'excelHtml5',
                text: '<i class="fas fa-file-excel"></i> Excel',
                title: 'Laporan Laba Rugi <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7]
                }
            },
            {
                extend: 'pdfHtml5',
                text: '<i class="fas fa-file-pdf"></i> PDF',
                title: 'Laporan Laba Rugi <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7]
                },
                customize: function(doc) {
                    doc.content[1].table.widths = 
                        Array(doc.content[1].table.body[0].length + 1).join('*').split('');
                }
            },
            {
                extend: 'print',
                text: '<i class="fas fa-print"></i> Print',
                title: 'Laporan Laba Rugi <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7]
                }
            }
        ],
        responsive: true,
        language: {
            url: '//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json'
        },
        order: [[1, 'desc']]
    });

    // Inisialisasi DataTables untuk kategori
    $('#dataTableKategori').DataTable({
        responsive: true,
        language: {
            url: '//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json'
        },
        order: [[3, 'desc']]
    });

    // Export PDF dengan parameter filter
    $('#btnExportPDF').click(function() {
        var params = new URLSearchParams(window.location.search);
        window.open('export_pdf_laba.php?' + params.toString(), '_blank');
    });
});
</script>

<?php include 'footer.php'; ?>