<?php
// Start session dan validasi login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Validasi hak akses (hanya Super Admin, Admin dan Owner)
$allowed_roles = ['0', '1', '3', '4']; // Sesuaikan dengan id_role super admin, admin, kasir dan owner
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

include 'header.php';
require_once 'config.php';

// Fungsi untuk format mata uang
function format_rupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

// Set default filter
$id_toko = isset($_GET['id_toko']) ? (int)$_GET['id_toko'] : 0;
$id_kategori = isset($_GET['id_kategori']) ? (int)$_GET['id_kategori'] : 0;
$status_stok = isset($_GET['status_stok']) ? $_GET['status_stok'] : '';

// Query untuk data laporan stok
$query = "SELECT b.id_barang, b.kode_barang, b.nama_barang, 
          k.nama_kategori, s.nama_satuan,
          st.stok_akhir, b.stok_minimal,
          b.harga_beli, b.harga_jual,
          (b.harga_jual - b.harga_beli) as laba_per_unit,
          (st.stok_akhir * b.harga_beli) as total_nilai_beli,
          (st.stok_akhir * b.harga_jual) as total_nilai_jual
          FROM barang b
          JOIN kategori k ON b.id_kategori = k.id_kategori
          JOIN satuan s ON b.id_satuan = s.id_satuan
          JOIN stok_toko st ON b.id_barang = st.id_barang
          WHERE b.is_active = 1";

$params = [];
$types = "";

// Filter berdasarkan toko
if ($id_toko > 0) {
    $query .= " AND st.id_toko = ?";
    $params[] = $id_toko;
    $types .= "i";
} elseif ($_SESSION['id_role'] != 0) { // Bukan Super Admin
    $query .= " AND st.id_toko = ?";
    $params[] = $_SESSION['id_toko'];
    $types .= "i";
}

// Filter berdasarkan kategori
if ($id_kategori > 0) {
    $query .= " AND b.id_kategori = ?";
    $params[] = $id_kategori;
    $types .= "i";
}

// Filter berdasarkan status stok
if (!empty($status_stok)) {
    if ($status_stok == 'hampir_habis') {
        $query .= " AND st.stok_akhir <= b.stok_minimal";
    } elseif ($status_stok == 'habis') {
        $query .= " AND st.stok_akhir = 0";
    } elseif ($status_stok == 'cukup') {
        $query .= " AND st.stok_akhir > b.stok_minimal";
    }
}

$query .= " ORDER BY k.nama_kategori, b.nama_barang";

// Prepare statement
$stmt = mysqli_prepare($conn, $query);
if ($stmt) {
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt, $types, ...$params);
    }
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
} else {
    die("Error in preparing statement: " . mysqli_error($conn));
}

// Hitung total nilai
$total_nilai_beli = 0;
$total_nilai_jual = 0;
$total_laba_potensial = 0;
$data_stok = [];
while ($row = mysqli_fetch_assoc($result)) {
    $data_stok[] = $row;
    $total_nilai_beli += $row['total_nilai_beli'];
    $total_nilai_jual += $row['total_nilai_jual'];
    $total_laba_potensial += ($row['total_nilai_jual'] - $row['total_nilai_beli']);
}

// Query untuk dropdown filter
$query_toko = "SELECT * FROM toko WHERE status_aktif = 1";
if ($_SESSION['id_role'] != 0) { // Bukan Super Admin
    $query_toko .= " AND id_toko = " . $_SESSION['id_toko'];
}
$result_toko = mysqli_query($conn, $query_toko);

$query_kategori = "SELECT * FROM kategori ORDER BY nama_kategori";
$result_kategori = mysqli_query($conn, $query_kategori);
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Laporan Stok Barang</h1>
    </div>

    <!-- Filter Form -->
    <div class="card mb-4 animate-fade-in">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary-custom">Filter Laporan</h6>
        </div>
        <div class="card-body">
            <form method="GET" action="laporan_stok.php">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="id_toko">Toko</label>
                            <select class="form-control" id="id_toko" name="id_toko" <?php echo ($_SESSION['id_role'] != 0) ? 'disabled' : ''; ?>>
                                <option value="0">Semua Toko</option>
                                <?php while ($toko = mysqli_fetch_assoc($result_toko)): ?>
                                    <option value="<?php echo $toko['id_toko']; ?>" 
                                        <?php echo ($id_toko == $toko['id_toko']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($toko['nama_toko']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <?php if ($_SESSION['id_role'] != 0): ?>
                                <input type="hidden" name="id_toko" value="<?php echo $_SESSION['id_toko']; ?>">
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="id_kategori">Kategori</label>
                            <select class="form-control" id="id_kategori" name="id_kategori">
                                <option value="0">Semua Kategori</option>
                                <?php while ($kategori = mysqli_fetch_assoc($result_kategori)): ?>
                                    <option value="<?php echo $kategori['id_kategori']; ?>" 
                                        <?php echo ($id_kategori == $kategori['id_kategori']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($kategori['nama_kategori']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="status_stok">Status Stok</label>
                            <select class="form-control" id="status_stok" name="status_stok">
                                <option value="">Semua Status</option>
                                <option value="hampir_habis" <?php echo ($status_stok == 'hampir_habis') ? 'selected' : ''; ?>>Hampir Habis</option>
                                <option value="habis" <?php echo ($status_stok == 'habis') ? 'selected' : ''; ?>>Habis</option>
                                <option value="cukup" <?php echo ($status_stok == 'cukup') ? 'selected' : ''; ?>>Cukup</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary-custom mr-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="laporan_stok.php" class="btn btn-secondary mr-2">
                            <i class="fas fa-sync-alt"></i> Reset
                        </a>
                        <button type="button" class="btn btn-success" id="btnExportPDF">
                            <i class="fas fa-file-pdf"></i> Export PDF
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Data Table -->
    <div class="card animate-fade-in">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary-custom">Data Stok Barang</h6>
            <div class="text-muted">
                <span class="badge bg-light text-dark">
                    <?php echo count($data_stok); ?> Barang
                </span>
                <span class="badge bg-info text-white ml-2">
                    Nilai Beli: <?php echo format_rupiah($total_nilai_beli); ?>
                </span>
                <span class="badge bg-success text-white ml-2">
                    Nilai Jual: <?php echo format_rupiah($total_nilai_jual); ?>
                </span>
                <span class="badge bg-warning text-dark ml-2">
                    Laba Potensial: <?php echo format_rupiah($total_laba_potensial); ?>
                </span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-hover" id="dataTable" width="100%" cellspacing="0">
                    <thead class="thead-light">
                        <tr>
                            <th>Kode</th>
                            <th>Nama Barang</th>
                            <th>Kategori</th>
                            <th>Satuan</th>
                            <th>Stok</th>
                            <th>Minimal</th>
                            <th>Status</th>
                            <th>Harga Beli</th>
                            <th>Harga Jual</th>
                            <th>Laba/Unit</th>
                            <th>Nilai Beli</th>
                            <th>Nilai Jual</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($data_stok as $stok): 
                            $status_class = '';
                            $status_text = '';
                            if ($stok['stok_akhir'] == 0) {
                                $status_class = 'danger';
                                $status_text = 'Habis';
                            } elseif ($stok['stok_akhir'] <= $stok['stok_minimal']) {
                                $status_class = 'warning';
                                $status_text = 'Hampir Habis';
                            } else {
                                $status_class = 'success';
                                $status_text = 'Cukup';
                            }
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stok['kode_barang']); ?></td>
                            <td><?php echo htmlspecialchars($stok['nama_barang']); ?></td>
                            <td><?php echo htmlspecialchars($stok['nama_kategori']); ?></td>
                            <td><?php echo htmlspecialchars($stok['nama_satuan']); ?></td>
                            <td class="text-center"><?php echo $stok['stok_akhir']; ?></td>
                            <td class="text-center"><?php echo $stok['stok_minimal']; ?></td>
                            <td class="text-center">
                                <span class="badge bg-<?php echo $status_class; ?>">
                                    <?php echo $status_text; ?>
                                </span>
                            </td>
                            <td class="text-right"><?php echo format_rupiah($stok['harga_beli']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($stok['harga_jual']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($stok['laba_per_unit']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($stok['total_nilai_beli']); ?></td>
                            <td class="text-right"><?php echo format_rupiah($stok['total_nilai_jual']); ?></td>
                            <td>
                                <a href="mutasi_stok.php?id_barang=<?php echo $stok['id_barang']; ?>" 
                                   class="btn btn-sm btn-primary-custom" title="Mutasi Stok">
                                    <i class="fas fa-exchange-alt"></i>
                                </a>
                                <a href="detail_barang.php?id=<?php echo $stok['id_barang']; ?>" 
                                   class="btn btn-sm btn-info" title="Detail">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr class="font-weight-bold">
                            <td colspan="10" class="text-right">TOTAL</td>
                            <td class="text-right"><?php echo format_rupiah($total_nilai_beli); ?></td>
                            <td class="text-right"><?php echo format_rupiah($total_nilai_jual); ?></td>
                            <td></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
</main>

<!-- Script untuk DataTables -->
<script>
$(document).ready(function() {
    // Inisialisasi DataTables
    $('#dataTable').DataTable({
        dom: 'Bfrtip',
        buttons: [
            {
                extend: 'excelHtml5',
                text: '<i class="fas fa-file-excel"></i> Excel',
                title: 'Laporan Stok Barang <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
                }
            },
            {
                extend: 'pdfHtml5',
                text: '<i class="fas fa-file-pdf"></i> PDF',
                title: 'Laporan Stok Barang <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
                },
                customize: function(doc) {
                    doc.content[1].table.widths = 
                        Array(doc.content[1].table.body[0].length + 1).join('*').split('');
                }
            },
            {
                extend: 'print',
                text: '<i class="fas fa-print"></i> Print',
                title: 'Laporan Stok Barang <?php echo date("d-m-Y"); ?>',
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
                }
            }
        ],
        responsive: true,
        language: {
            url: '//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json'
        },
        order: [[2, 'asc'], [1, 'asc']] // Urutkan berdasarkan kategori dan nama barang
    });

    // Export PDF dengan parameter filter
    $('#btnExportPDF').click(function() {
        var params = new URLSearchParams(window.location.search);
        window.open('export_pdf_stok.php?' + params.toString(), '_blank');
    });
});
</script>

<?php include 'footer.php'; ?>