<?php
// File: lupa_password.php
require_once 'config.php';
session_start();

// Redirect jika sudah login
if (isset($_SESSION['id_pengguna'])) {
    header("Location: dashboard.php");
    exit();
}

$error = '';
$success = '';
$show_form = true;
$password_reset_success = false; // Tambahkan variabel untuk menandai sukses reset password

// Proses reset password
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['request_reset'])) {
        // Tahap 1: Request reset password
        $email = clean_input($_POST['email']);
        
        if (empty($email)) {
            $error = "Email harus diisi!";
        } else {
            // Cek apakah email ada di database
            $stmt = $conn->prepare("SELECT id_pengguna, nama_lengkap FROM pengguna WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows == 1) {
                $user = $result->fetch_assoc();
                
                // Generate token reset password
                $token = bin2hex(random_bytes(32));
                $expires = date("Y-m-d H:i:s", strtotime("+1 hour")); // Token berlaku 1 jam
                
                // Simpan token ke database
                $update_stmt = $conn->prepare("UPDATE pengguna SET reset_token = ?, reset_token_expires = ? WHERE id_pengguna = ?");
                $update_stmt->bind_param("ssi", $token, $expires, $user['id_pengguna']);
                
                if ($update_stmt->execute()) {
                    // Kirim email reset password (implementasi sederhana)
                    $reset_link = "http://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . "/lupa_password.php?token=$token";
                    
                    // Ini hanya simulasi, di production gunakan PHPMailer atau library email
                    $email_sent = true; // Asumsi email terkirim
                    
                    if ($email_sent) {
                        $success = "Link Reset Password Telah Dikirim! Silakan Cek Kotak Masuk (Inbox) Atau Folder Spam Pada Email Anda. Jika Anda Tidak Menerima Email Dalam Beberapa Menit, Harap Hubungi Administrator atau Tim IT / GIS HO Untuk Bantuan Lebih Lanjut.";
                        $show_form = false;
                    } else {
                        $error = "Gagal mengirim email reset password. Silakan coba lagi.";
                    }
                } else {
                    $error = "Gagal membuat token reset password. Silakan coba lagi.";
                }
            } else {
                $error = "Email tidak terdaftar dalam sistem.";
            }
        }
    } elseif (isset($_POST['reset_password'])) {
        // Tahap 2: Reset password dengan token
        $token = clean_input($_POST['token']);
        $password = clean_input($_POST['password']);
        $confirm_password = clean_input($_POST['confirm_password']);
        
        if (empty($password) || empty($confirm_password)) {
            $error = "Password dan konfirmasi password harus diisi!";
        } elseif ($password !== $confirm_password) {
            $error = "Password dan konfirmasi password tidak sama!";
        } elseif (strlen($password) < 8) {
            $error = "Password minimal 8 karakter!";
        } else {
            // Cek token valid
            $current_time = date("Y-m-d H:i:s");
            $stmt = $conn->prepare("SELECT id_pengguna FROM pengguna WHERE reset_token = ? AND reset_token_expires > ?");
            $stmt->bind_param("ss", $token, $current_time);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows == 1) {
                $user = $result->fetch_assoc();
                
                // Hash password baru
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                // Update password dan hapus token
                $update_stmt = $conn->prepare("UPDATE pengguna SET password = ?, reset_token = NULL, reset_token_expires = NULL WHERE id_pengguna = ?");
                $update_stmt->bind_param("si", $hashed_password, $user['id_pengguna']);
                
                if ($update_stmt->execute()) {
                    $success = "Password berhasil direset. Silakan login dengan password baru Anda.";
                    $show_form = false;
                    $password_reset_success = true; // Set flag sukses reset password
                } else {
                    $error = "Gagal mereset password. Silakan coba lagi.";
                }
            } else {
                $error = "Token reset password tidak valid atau sudah kadaluarsa. Silakan request reset password lagi.";
                $show_form = true;
            }
        }
    }
}

// Jika ada token di URL (user mengklik link dari email)
if (isset($_GET['token'])) {
    $token = clean_input($_GET['token']);
    
    // Cek token valid
    $current_time = date("Y-m-d H:i:s");
    $stmt = $conn->prepare("SELECT id_pengguna FROM pengguna WHERE reset_token = ? AND reset_token_expires > ?");
    $stmt->bind_param("ss", $token, $current_time);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows == 1) {
        $show_form = false; // Tampilkan form reset password
    } else {
        $error = "Token reset password tidak valid atau sudah kadaluarsa. Silakan request reset password lagi.";
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Lupa Password - Sistem Manajemen Toko Kuliner</title>
    
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary: #2e7d32; /* Hijau */
            --primary-dark: #1b5e20; /* Hijau tua */
            --secondary: #d32f2f; /* Merah */
            --light: #f8f9fa;
            --dark: #212121;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            height: 100vh;
            display: flex;
            align-items: center;
            background: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), 
                        url('assets/bg.jpg') no-repeat center center fixed;
            background-size: cover;
        }
        
        .reset-container {
            max-width: 450px;
            width: 100%;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
            overflow: hidden;
        }
        
        .reset-header {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
            padding: 25px;
            text-align: center;
        }
        
        .reset-header h2 {
            margin: 0;
            font-weight: 700;
            font-size: 1.8rem;
        }
        
        .reset-body {
            padding: 30px;
        }
        
        .btn-reset {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            height: 50px;
            border-radius: 8px;
            font-weight: 600;
            border: none;
            width: 100%;
            transition: all 0.3s;
        }
        
        .btn-reset:hover {
            background: linear-gradient(to right, var(--primary-dark), var(--primary));
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .btn-login {
            background: linear-gradient(to right, #4e73df, #224abe);
            color: white;
            height: 50px;
            border-radius: 8px;
            font-weight: 600;
            border: none;
            width: 100%;
            transition: all 0.3s;
            margin-bottom: 10px;
        }
        
        .btn-login:hover {
            background: linear-gradient(to right, #224abe, #4e73df);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .btn-register {
            background: linear-gradient(to right, #1cc88a, #13855c);
            color: white;
            height: 50px;
            border-radius: 8px;
            font-weight: 600;
            border: none;
            width: 100%;
            transition: all 0.3s;
        }
        
        .btn-register:hover {
            background: linear-gradient(to right, #13855c, #1cc88a);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .alert {
            border-radius: 8px;
        }
        
        .password-strength {
            height: 5px;
            margin-top: 5px;
            background-color: #e9ecef;
            border-radius: 3px;
            overflow: hidden;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            transition: width 0.3s;
        }
        
        .success-message {
            text-align: center;
            padding: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="reset-container">
            <div class="reset-header">
                <h2>Reset Password</h2>
                <p>Sistem Manajemen Toko</p>
            </div>
            
            <div class="reset-body">
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-circle me-2"></i> <?= $error ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle me-2"></i> <?= $success ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($password_reset_success): ?>
                    <!-- Tampilkan pesan sukses dan tombol setelah reset password berhasil -->
                    <div class="success-message">
                        <h4><i class="fas fa-check-circle text-success mb-3"></i> Password Anda Telah Berhasil Direset!</h4>
                        <p class="mb-4">Silakan login dengan password baru Anda.</p>
                        
                        <div class="d-grid gap-2">
                            <a href="login.php" class="btn btn-login">
                                <i class="fas fa-sign-in-alt me-2"></i> KEMBALI KE LOGIN
                            </a>
                            <a href="register.php" class="btn btn-register">
                                <i class="fas fa-user-plus me-2"></i> BUAT AKUN BARU
                            </a>
                        </div>
                    </div>
                <?php elseif ($show_form && !isset($_GET['token'])): ?>
                    <!-- Form Request Reset Password -->
                    <form method="POST" action="">
                        <div class="mb-4">
                            <p>Masukkan alamat email yang terdaftar. Kami akan mengirimkan link untuk mereset password Anda.</p>
                        </div>
                        
                        <div class="mb-4">
                            <label for="email" class="form-label fw-bold">Email</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-envelope"></i></span>
                                <input type="email" class="form-control" id="email" name="email" placeholder="Masukkan email terdaftar" required>
                            </div>
                        </div>
                        
                        <button type="submit" name="request_reset" class="btn btn-reset mb-4">
                            <i class="fas fa-paper-plane me-2"></i> KIRIM LINK RESET
                        </button>
                        
                        <div class="text-center pt-3 border-top">
                            <p class="mb-0">Ingat password? <a href="login.php" class="fw-bold">Login disini</a></p>
                        </div>
                    </form>
                
                <?php elseif (!$show_form && isset($_GET['token'])): ?>
                    <!-- Form Reset Password dengan Token -->
                    <form method="POST" action="">
                        <input type="hidden" name="token" value="<?= htmlspecialchars($_GET['token']) ?>">
                        
                        <div class="mb-4">
                            <label for="password" class="form-label fw-bold">Password Baru</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                <input type="password" class="form-control" id="password" name="password" placeholder="Masukkan password baru" required>
                                <button class="btn btn-outline-secondary toggle-password" type="button">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="password-strength mt-2">
                                <div class="password-strength-bar" id="password-strength-bar"></div>
                            </div>
                            <small class="text-muted">Minimal 8 karakter</small>
                        </div>
                        
                        <div class="mb-4">
                            <label for="confirm_password" class="form-label fw-bold">Konfirmasi Password</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" placeholder="Masukkan ulang password" required>
                                <button class="btn btn-outline-secondary toggle-password" type="button">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        
                        <button type="submit" name="reset_password" class="btn btn-reset mb-4">
                            <i class="fas fa-sync-alt me-2"></i> RESET PASSWORD
                        </button>
                        
                        <div class="text-center pt-3 border-top">
                            <p class="mb-0">Ingat password? <a href="login.php" class="fw-bold">Login disini</a></p>
                        </div>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Toggle password visibility
        document.querySelectorAll('.toggle-password').forEach(button => {
            button.addEventListener('click', function() {
                const passwordInput = this.parentElement.querySelector('input');
                const icon = this.querySelector('i');
                
                if (passwordInput.type === 'password') {
                    passwordInput.type = 'text';
                    icon.className = 'fas fa-eye-slash';
                } else {
                    passwordInput.type = 'password';
                    icon.className = 'fas fa-eye';
                }
            });
        });
        
        // Password strength indicator
        document.getElementById('password')?.addEventListener('input', function() {
            const password = this.value;
            const strengthBar = document.getElementById('password-strength-bar');
            let strength = 0;
            
            // Check length
            if (password.length >= 8) strength += 1;
            if (password.length >= 12) strength += 1;
            
            // Check for numbers
            if (password.match(/\d+/)) strength += 1;
            
            // Check for special chars
            if (password.match(/[!@#$%^&*(),.?":{}|<>]/)) strength += 1;
            
            // Check for uppercase and lowercase
            if (password.match(/[a-z]/) && password.match(/[A-Z]/)) strength += 1;
            
            // Update strength bar
            switch(strength) {
                case 0:
                    strengthBar.style.width = '0%';
                    strengthBar.style.backgroundColor = '#dc3545'; // Red
                    break;
                case 1:
                    strengthBar.style.width = '20%';
                    strengthBar.style.backgroundColor = '#dc3545'; // Red
                    break;
                case 2:
                    strengthBar.style.width = '40%';
                    strengthBar.style.backgroundColor = '#fd7e14'; // Orange
                    break;
                case 3:
                    strengthBar.style.width = '60%';
                    strengthBar.style.backgroundColor = '#ffc107'; // Yellow
                    break;
                case 4:
                    strengthBar.style.width = '80%';
                    strengthBar.style.backgroundColor = '#28a745'; // Green
                    break;
                case 5:
                    strengthBar.style.width = '100%';
                    strengthBar.style.backgroundColor = '#28a745'; // Green
                    break;
            }
        });
    </script>
</body>
</html>