<?php
session_start();
require_once 'config.php';

// Cek apakah user sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Fungsi untuk format mata uang
function format_rupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

// Fungsi untuk generate nomor transaksi
function generate_no_transaksi() {
    global $conn;
    $prefix = "PBL-" . date('ymd') . "-";
    $query = "SELECT MAX(no_faktur) as last_number FROM pembelian WHERE no_faktur LIKE '$prefix%'";
    $result = mysqli_query($conn, $query);
    $row = mysqli_fetch_assoc($result);
    
    if ($row['last_number']) {
        $last_number = intval(substr($row['last_number'], -4));
        $new_number = $last_number + 1;
    } else {
        $new_number = 1;
    }
    
    return $prefix . str_pad($new_number, 4, '0', STR_PAD_LEFT);
}

// Proses tambah pembelian
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['tambah_pembelian'])) {
        $no_faktur = $_POST['no_faktur'];
        $id_supplier = $_POST['id_supplier'];
        $id_toko = $_POST['id_toko'];
        $tanggal_pembelian = $_POST['tanggal_pembelian'];
        $id_metode = $_POST['id_metode'];
        $total_pembelian = str_replace('.', '', $_POST['total_pembelian']);
        $diskon = str_replace('.', '', $_POST['diskon']);
        $ppn = str_replace('.', '', $_POST['ppn']);
        $grand_total = str_replace('.', '', $_POST['grand_total']);
        $jumlah_bayar = str_replace('.', '', $_POST['jumlah_bayar']);
        $kembalian = str_replace('.', '', $_POST['kembalian']);
        $catatan = $_POST['catatan'];
        $id_pengguna = $_SESSION['id_pengguna'];
        
        // Hitung sisa hutang
        $sisa_hutang = $grand_total - $jumlah_bayar;
        
        // Tentukan status pembayaran
        if ($sisa_hutang <= 0) {
            $status_pembayaran = 'lunas';
            $sisa_hutang = 0;
        } elseif ($jumlah_bayar > 0) {
            $status_pembayaran = 'sebagian';
        } else {
            $status_pembayaran = 'hutang';
        }
        
        // Mulai transaksi
        mysqli_begin_transaction($conn);
        
        try {
            // Insert ke tabel pembelian
            $query = "INSERT INTO pembelian (no_faktur, id_supplier, id_toko, id_pengguna, tanggal_pembelian, total_pembelian, diskon, ppn, grand_total, status_pembayaran, jumlah_bayar, sisa_hutang, catatan, id_metode) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, 'siiisddddssdsi', $no_faktur, $id_supplier, $id_toko, $id_pengguna, $tanggal_pembelian, $total_pembelian, $diskon, $ppn, $grand_total, $status_pembayaran, $jumlah_bayar, $sisa_hutang, $catatan, $id_metode);
            mysqli_stmt_execute($stmt);
            $id_pembelian = mysqli_insert_id($conn);
            
            // Insert detail pembelian dan update stok
            foreach ($_POST['id_barang'] as $key => $id_barang) {
                $jumlah = $_POST['jumlah'][$key];
                $harga_beli = str_replace('.', '', $_POST['harga_beli'][$key]);
                $subtotal = str_replace('.', '', $_POST['subtotal'][$key]);
                $expired_date = $_POST['expired_date'][$key] ?? null;
                
                // Insert detail pembelian
                $query_detail = "INSERT INTO detail_pembelian (id_pembelian, id_barang, jumlah, harga_beli, subtotal, expired_date) 
                                 VALUES (?, ?, ?, ?, ?, ?)";
                $stmt_detail = mysqli_prepare($conn, $query_detail);
                mysqli_stmt_bind_param($stmt_detail, 'iiidds', $id_pembelian, $id_barang, $jumlah, $harga_beli, $subtotal, $expired_date);
                mysqli_stmt_execute($stmt_detail);
                
                // Update stok toko
                $query_stok = "INSERT INTO stok_toko (id_barang, id_toko, stok_masuk, stok_akhir, expired_date) 
                               VALUES (?, ?, ?, ?, ?)
                               ON DUPLICATE KEY UPDATE 
                               stok_masuk = stok_masuk + VALUES(stok_masuk),
                               stok_akhir = stok_akhir + VALUES(stok_masuk),
                               expired_date = IF(VALUES(expired_date) IS NOT NULL AND (expired_date IS NULL OR VALUES(expired_date) < expired_date), VALUES(expired_date), expired_date)";
                $stmt_stok = mysqli_prepare($conn, $query_stok);
                mysqli_stmt_bind_param($stmt_stok, 'iiiis', $id_barang, $id_toko, $jumlah, $jumlah, $expired_date);
                mysqli_stmt_execute($stmt_stok);
            }
            
            // Jika ada hutang, tambahkan ke tabel hutang_supplier
            if ($status_pembayaran != 'lunas') {
                $tanggal_jatuh_tempo = date('Y-m-d', strtotime($tanggal_pembelian . ' + 30 days'));
                
                $query_hutang = "INSERT INTO hutang_supplier (id_pembelian, id_supplier, tanggal_hutang, tanggal_jatuh_tempo, jumlah_hutang, jumlah_bayar, sisa_hutang, status) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt_hutang = mysqli_prepare($conn, $query_hutang);
                mysqli_stmt_bind_param($stmt_hutang, 'iissddds', $id_pembelian, $id_supplier, $tanggal_pembelian, $tanggal_jatuh_tempo, $grand_total, $jumlah_bayar, $sisa_hutang, $status_pembayaran);
                mysqli_stmt_execute($stmt_hutang);
            }
            
            // Proses upload nota jika ada
            if (!empty($_FILES['nota']['name'])) {
                $target_dir = "uploads/nota_pembelian/";
                if (!file_exists($target_dir)) {
                    mkdir($target_dir, 0777, true);
                }
                
                $file_name = basename($_FILES['nota']['name']);
                $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                $new_file_name = "nota_" . $id_pembelian . "_" . time() . "." . $file_ext;
                $target_file = $target_dir . $new_file_name;
                
                if (move_uploaded_file($_FILES['nota']['tmp_name'], $target_file)) {
                    $query_nota = "INSERT INTO nota_pembelian (id_pembelian, nama_file, path_file) VALUES (?, ?, ?)";
                    $stmt_nota = mysqli_prepare($conn, $query_nota);
                    mysqli_stmt_bind_param($stmt_nota, 'iss', $id_pembelian, $file_name, $target_file);
                    mysqli_stmt_execute($stmt_nota);
                }
            }
            
            // Commit transaksi
            mysqli_commit($conn);
            
            $_SESSION['success'] = "Transaksi pembelian berhasil disimpan";
            header("Location: pembelian.php");
            exit();
        } catch (Exception $e) {
            // Rollback transaksi jika ada error
            mysqli_rollback($conn);
            $_SESSION['error'] = "Gagal menyimpan transaksi: " . $e->getMessage();
            header("Location: pembelian.php");
            exit();
        }
    }
}

// Proses update pembelian
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['update_pembelian'])) {
    $id_pembelian = $_POST['id_pembelian'];
    $id_supplier = $_POST['id_supplier'];
    $id_toko = $_POST['id_toko'];
    $tanggal_pembelian = $_POST['tanggal_pembelian'];
    $id_metode = $_POST['id_metode'];
    $total_pembelian = str_replace('.', '', $_POST['total_pembelian']);
    $diskon = str_replace('.', '', $_POST['diskon']);
    $ppn = str_replace('.', '', $_POST['ppn']);
    $grand_total = str_replace('.', '', $_POST['grand_total']);
    $jumlah_bayar = str_replace('.', '', $_POST['jumlah_bayar']);
    $kembalian = str_replace('.', '', $_POST['kembalian']);
    $catatan = $_POST['catatan'];
    
    // Hitung sisa hutang
    $sisa_hutang = $grand_total - $jumlah_bayar;
    
    // Tentukan status pembayaran
    if ($sisa_hutang <= 0) {
        $status_pembayaran = 'lunas';
        $sisa_hutang = 0;
    } elseif ($jumlah_bayar > 0) {
        $status_pembayaran = 'sebagian';
    } else {
        $status_pembayaran = 'hutang';
    }
    
    // Mulai transaksi
    mysqli_begin_transaction($conn);
    
    try {
        // Update data pembelian
        $query = "UPDATE pembelian SET 
                  id_supplier = ?, 
                  id_toko = ?, 
                  tanggal_pembelian = ?, 
                  id_metode = ?,
                  total_pembelian = ?, 
                  diskon = ?, 
                  ppn = ?, 
                  grand_total = ?, 
                  status_pembayaran = ?, 
                  jumlah_bayar = ?, 
                  sisa_hutang = ?, 
                  catatan = ? 
                  WHERE id_pembelian = ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, 'iisddddssdsi', $id_supplier, $id_toko, $tanggal_pembelian, $id_metode, $total_pembelian, $diskon, $ppn, $grand_total, $status_pembayaran, $jumlah_bayar, $sisa_hutang, $catatan, $id_pembelian);
        mysqli_stmt_execute($stmt);
        
        // Hapus detail pembelian lama dan kembalikan stok
        $query_detail_lama = "SELECT id_barang, jumlah FROM detail_pembelian WHERE id_pembelian = ?";
        $stmt_detail_lama = mysqli_prepare($conn, $query_detail_lama);
        mysqli_stmt_bind_param($stmt_detail_lama, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_detail_lama);
        $result_detail_lama = mysqli_stmt_get_result($stmt_detail_lama);
        
        while ($detail = mysqli_fetch_assoc($result_detail_lama)) {
            $query_kembalikan_stok = "UPDATE stok_toko 
                                     SET stok_masuk = stok_masuk - ?, 
                                         stok_akhir = stok_akhir - ? 
                                     WHERE id_barang = ? AND id_toko = ?";
            $stmt_kembalikan_stok = mysqli_prepare($conn, $query_kembalikan_stok);
            mysqli_stmt_bind_param($stmt_kembalikan_stok, 'iiii', $detail['jumlah'], $detail['jumlah'], $detail['id_barang'], $id_toko);
            mysqli_stmt_execute($stmt_kembalikan_stok);
        }
        
        // Hapus detail pembelian lama
        $query_hapus_detail = "DELETE FROM detail_pembelian WHERE id_pembelian = ?";
        $stmt_hapus_detail = mysqli_prepare($conn, $query_hapus_detail);
        mysqli_stmt_bind_param($stmt_hapus_detail, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_hapus_detail);
        
        // Insert detail pembelian baru dan update stok
        foreach ($_POST['id_barang'] as $key => $id_barang) {
            $jumlah = $_POST['jumlah'][$key];
            $harga_beli = str_replace('.', '', $_POST['harga_beli'][$key]);
            $subtotal = str_replace('.', '', $_POST['subtotal'][$key]);
            $expired_date = $_POST['expired_date'][$key] ?? null;
            
            // Insert detail pembelian baru
            $query_detail = "INSERT INTO detail_pembelian (id_pembelian, id_barang, jumlah, harga_beli, subtotal, expired_date) 
                             VALUES (?, ?, ?, ?, ?, ?)";
            $stmt_detail = mysqli_prepare($conn, $query_detail);
            mysqli_stmt_bind_param($stmt_detail, 'iiidds', $id_pembelian, $id_barang, $jumlah, $harga_beli, $subtotal, $expired_date);
            mysqli_stmt_execute($stmt_detail);
            
            // Update stok toko
            $query_stok = "UPDATE stok_toko 
                           SET stok_masuk = stok_masuk + ?, 
                               stok_akhir = stok_akhir + ?,
                               expired_date = IF(? IS NOT NULL AND (expired_date IS NULL OR ? < expired_date), ?, expired_date)
                           WHERE id_barang = ? AND id_toko = ?";
            $stmt_stok = mysqli_prepare($conn, $query_stok);
            mysqli_stmt_bind_param($stmt_stok, 'iissii', $jumlah, $jumlah, $expired_date, $expired_date, $expired_date, $id_barang, $id_toko);
            mysqli_stmt_execute($stmt_stok);
        }
        
        // Update hutang supplier
        if ($status_pembayaran != 'lunas') {
            $tanggal_jatuh_tempo = date('Y-m-d', strtotime($tanggal_pembelian . ' + 30 days'));
            
            // Cek apakah sudah ada hutang untuk pembelian ini
            $query_cek_hutang = "SELECT id_hutang_supplier FROM hutang_supplier WHERE id_pembelian = ?";
            $stmt_cek_hutang = mysqli_prepare($conn, $query_cek_hutang);
            mysqli_stmt_bind_param($stmt_cek_hutang, 'i', $id_pembelian);
            mysqli_stmt_execute($stmt_cek_hutang);
            $result_cek_hutang = mysqli_stmt_get_result($stmt_cek_hutang);
            
            if (mysqli_num_rows($result_cek_hutang) > 0) {
                // Update hutang yang sudah ada
                $query_update_hutang = "UPDATE hutang_supplier SET 
                                        id_supplier = ?, 
                                        tanggal_hutang = ?, 
                                        tanggal_jatuh_tempo = ?, 
                                        jumlah_hutang = ?, 
                                        jumlah_bayar = ?, 
                                        sisa_hutang = ?, 
                                        status = ? 
                                        WHERE id_pembelian = ?";
                $stmt_update_hutang = mysqli_prepare($conn, $query_update_hutang);
                mysqli_stmt_bind_param($stmt_update_hutang, 'issdddss', $id_supplier, $tanggal_pembelian, $tanggal_jatuh_tempo, $grand_total, $jumlah_bayar, $sisa_hutang, $status_pembayaran, $id_pembelian);
                mysqli_stmt_execute($stmt_update_hutang);
            } else {
                // Insert hutang baru
                $query_insert_hutang = "INSERT INTO hutang_supplier (id_pembelian, id_supplier, tanggal_hutang, tanggal_jatuh_tempo, jumlah_hutang, jumlah_bayar, sisa_hutang, status) 
                                         VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt_insert_hutang = mysqli_prepare($conn, $query_insert_hutang);
                mysqli_stmt_bind_param($stmt_insert_hutang, 'iissddds', $id_pembelian, $id_supplier, $tanggal_pembelian, $tanggal_jatuh_tempo, $grand_total, $jumlah_bayar, $sisa_hutang, $status_pembayaran);
                mysqli_stmt_execute($stmt_insert_hutang);
            }
        } else {
            // Jika status lunas, hapus hutang jika ada
            $query_hapus_hutang = "DELETE FROM hutang_supplier WHERE id_pembelian = ?";
            $stmt_hapus_hutang = mysqli_prepare($conn, $query_hapus_hutang);
            mysqli_stmt_bind_param($stmt_hapus_hutang, 'i', $id_pembelian);
            mysqli_stmt_execute($stmt_hapus_hutang);
        }
        
        // Proses upload nota jika ada
        if (!empty($_FILES['nota']['name'])) {
            $target_dir = "uploads/nota_pembelian/";
            if (!file_exists($target_dir)) {
                mkdir($target_dir, 0777, true);
            }
            
            // Hapus nota lama jika ada
            $query_hapus_nota = "DELETE FROM nota_pembelian WHERE id_pembelian = ?";
            $stmt_hapus_nota = mysqli_prepare($conn, $query_hapus_nota);
            mysqli_stmt_bind_param($stmt_hapus_nota, 'i', $id_pembelian);
            mysqli_stmt_execute($stmt_hapus_nota);
            
            // Upload nota baru
            $file_name = basename($_FILES['nota']['name']);
            $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            $new_file_name = "nota_" . $id_pembelian . "_" . time() . "." . $file_ext;
            $target_file = $target_dir . $new_file_name;
            
            if (move_uploaded_file($_FILES['nota']['tmp_name'], $target_file)) {
                $query_nota = "INSERT INTO nota_pembelian (id_pembelian, nama_file, path_file) VALUES (?, ?, ?)";
                $stmt_nota = mysqli_prepare($conn, $query_nota);
                mysqli_stmt_bind_param($stmt_nota, 'iss', $id_pembelian, $file_name, $target_file);
                mysqli_stmt_execute($stmt_nota);
            }
        }
        
        // Commit transaksi
        mysqli_commit($conn);
        
        $_SESSION['success'] = "Transaksi pembelian berhasil diperbarui";
        header("Location: pembelian.php");
        exit();
    } catch (Exception $e) {
        // Rollback transaksi jika ada error
        mysqli_rollback($conn);
        $_SESSION['error'] = "Gagal memperbarui transaksi: " . $e->getMessage();
        header("Location: pembelian.php");
        exit();
    }
}

// Proses hapus pembelian
if (isset($_GET['hapus'])) {
    $id_pembelian = $_GET['hapus'];
    
    // Mulai transaksi
    mysqli_begin_transaction($conn);
    
    try {
        // Ambil data pembelian untuk rollback stok
        $query_pembelian = "SELECT id_toko FROM pembelian WHERE id_pembelian = ?";
        $stmt_pembelian = mysqli_prepare($conn, $query_pembelian);
        mysqli_stmt_bind_param($stmt_pembelian, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_pembelian);
        $result_pembelian = mysqli_stmt_get_result($stmt_pembelian);
        $pembelian = mysqli_fetch_assoc($result_pembelian);
        $id_toko = $pembelian['id_toko'];
        
        // Ambil detail pembelian untuk rollback stok
        $query_detail = "SELECT id_barang, jumlah FROM detail_pembelian WHERE id_pembelian = ?";
        $stmt_detail = mysqli_prepare($conn, $query_detail);
        mysqli_stmt_bind_param($stmt_detail, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_detail);
        $result_detail = mysqli_stmt_get_result($stmt_detail);
        
        // Rollback stok
        while ($detail = mysqli_fetch_assoc($result_detail)) {
            $query_stok = "UPDATE stok_toko 
                           SET stok_masuk = stok_masuk - ?, 
                               stok_akhir = stok_akhir - ? 
                           WHERE id_barang = ? AND id_toko = ?";
            $stmt_stok = mysqli_prepare($conn, $query_stok);
            mysqli_stmt_bind_param($stmt_stok, 'iiii', $detail['jumlah'], $detail['jumlah'], $detail['id_barang'], $id_toko);
            mysqli_stmt_execute($stmt_stok);
        }
        
        // Hapus nota pembelian jika ada
        $query_hapus_nota = "DELETE FROM nota_pembelian WHERE id_pembelian = ?";
        $stmt_hapus_nota = mysqli_prepare($conn, $query_hapus_nota);
        mysqli_stmt_bind_param($stmt_hapus_nota, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_hapus_nota);
        
        // Hapus dari hutang_supplier jika ada
        $query_hutang = "DELETE FROM hutang_supplier WHERE id_pembelian = ?";
        $stmt_hutang = mysqli_prepare($conn, $query_hutang);
        mysqli_stmt_bind_param($stmt_hutang, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_hutang);
        
        // Hapus detail pembelian
        $query_delete_detail = "DELETE FROM detail_pembelian WHERE id_pembelian = ?";
        $stmt_delete_detail = mysqli_prepare($conn, $query_delete_detail);
        mysqli_stmt_bind_param($stmt_delete_detail, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_delete_detail);
        
        // Hapus pembelian
        $query_delete = "DELETE FROM pembelian WHERE id_pembelian = ?";
        $stmt_delete = mysqli_prepare($conn, $query_delete);
        mysqli_stmt_bind_param($stmt_delete, 'i', $id_pembelian);
        mysqli_stmt_execute($stmt_delete);
        
        // Commit transaksi
        mysqli_commit($conn);
        
        $_SESSION['success'] = "Transaksi pembelian berhasil dihapus";
        header("Location: pembelian.php");
        exit();
    } catch (Exception $e) {
        // Rollback transaksi jika ada error
        mysqli_rollback($conn);
        $_SESSION['error'] = "Gagal menghapus transaksi: " . $e->getMessage();
        header("Location: pembelian.php");
        exit();
    }
}

// Ambil data pembelian untuk ditampilkan
$where = "1=1";
$params = [];

// Filter pencarian
if (isset($_GET['filter'])) {
    if (!empty($_GET['dari_tanggal'])) {
        $where .= " AND p.tanggal_pembelian >= ?";
        $params[] = $_GET['dari_tanggal'];
    }
    
    if (!empty($_GET['sampai_tanggal'])) {
        $where .= " AND p.tanggal_pembelian <= ?";
        $params[] = $_GET['sampai_tanggal'] . ' 23:59:59';
    }
    
    if (!empty($_GET['id_supplier'])) {
        $where .= " AND p.id_supplier = ?";
        $params[] = $_GET['id_supplier'];
    }
    
    if (!empty($_GET['status_pembayaran'])) {
        $where .= " AND p.status_pembayaran = ?";
        $params[] = $_GET['status_pembayaran'];
    }
    
    if (!empty($_GET['id_toko'])) {
        $where .= " AND p.id_toko = ?";
        $params[] = $_GET['id_toko'];
    }
}

// Query data pembelian
$query = "SELECT p.*, s.nama_supplier, t.nama_toko, m.nama_metode, pg.nama_lengkap as nama_kasir 
          FROM pembelian p
          JOIN supplier s ON p.id_supplier = s.id_supplier
          JOIN toko t ON p.id_toko = t.id_toko
          JOIN metode_pembayaran m ON p.id_metode = m.id_metode
          JOIN pengguna pg ON p.id_pengguna = pg.id_pengguna
          WHERE $where
          ORDER BY p.tanggal_pembelian DESC";
$stmt = mysqli_prepare($conn, $query);

if (!empty($params)) {
    $types = str_repeat('s', count($params));
    mysqli_stmt_bind_param($stmt, $types, ...$params);
}

mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$pembelian = mysqli_fetch_all($result, MYSQLI_ASSOC);

// Ambil data supplier untuk dropdown
$query_supplier = "SELECT id_supplier, nama_supplier FROM supplier WHERE status_aktif = 1 ORDER BY nama_supplier";
$result_supplier = mysqli_query($conn, $query_supplier);
$suppliers = mysqli_fetch_all($result_supplier, MYSQLI_ASSOC);

// Ambil data toko untuk dropdown
$query_toko = "SELECT id_toko, nama_toko FROM toko WHERE status_aktif = 1 ORDER BY nama_toko";
$result_toko = mysqli_query($conn, $query_toko);
$toko_list = mysqli_fetch_all($result_toko, MYSQLI_ASSOC);

// Ambil data metode pembayaran untuk dropdown
$query_metode = "SELECT id_metode, nama_metode FROM metode_pembayaran WHERE is_active = 1 ORDER BY nama_metode";
$result_metode = mysqli_query($conn, $query_metode);
$metode_list = mysqli_fetch_all($result_metode, MYSQLI_ASSOC);

// Ambil data untuk modal edit
$edit_data = null;
if (isset($_GET['edit'])) {
    $id_pembelian = $_GET['edit'];
    
    $query_edit = "SELECT p.*, np.nama_file 
                   FROM pembelian p
                   LEFT JOIN nota_pembelian np ON p.id_pembelian = np.id_pembelian
                   WHERE p.id_pembelian = ?";
    $stmt_edit = mysqli_prepare($conn, $query_edit);
    mysqli_stmt_bind_param($stmt_edit, 'i', $id_pembelian);
    mysqli_stmt_execute($stmt_edit);
    $result_edit = mysqli_stmt_get_result($stmt_edit);
    $edit_data = mysqli_fetch_assoc($result_edit);
    
    // Ambil detail pembelian untuk edit
    $query_detail = "SELECT dp.*, b.kode_barang, b.nama_barang 
                     FROM detail_pembelian dp
                     JOIN barang b ON dp.id_barang = b.id_barang
                     WHERE dp.id_pembelian = ?";
    $stmt_detail = mysqli_prepare($conn, $query_detail);
    mysqli_stmt_bind_param($stmt_detail, 'i', $id_pembelian);
    mysqli_stmt_execute($stmt_detail);
    $result_detail = mysqli_stmt_get_result($stmt_detail);
    $edit_detail = mysqli_fetch_all($result_detail, MYSQLI_ASSOC);
}

include 'header.php';
?>

<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Transaksi Pembelian</h1>
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
            <li class="breadcrumb-item">Transaksi</li>
            <li class="breadcrumb-item active" aria-current="page">Pembelian</li>
        </ol>
        <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#tambahPembelianModal">
            <i class="fas fa-plus"></i> Tambah Pembelian
        </button>
    </div>

    <!-- Filter Pencarian -->
    <div class="card mb-4 animate-fade-in">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Filter Pencarian</h6>
        </div>
        <div class="card-body">
            <form method="GET" action="pembelian.php">
                <div class="row">
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="dari_tanggal">Dari Tanggal</label>
                            <input type="date" class="form-control" id="dari_tanggal" name="dari_tanggal" value="<?= $_GET['dari_tanggal'] ?? '' ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="sampai_tanggal">Sampai Tanggal</label>
                            <input type="date" class="form-control" id="sampai_tanggal" name="sampai_tanggal" value="<?= $_GET['sampai_tanggal'] ?? '' ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="id_supplier">Supplier</label>
                            <select class="form-control" id="id_supplier" name="id_supplier">
                                <option value="">Semua Supplier</option>
                                <?php foreach ($suppliers as $supplier): ?>
                                    <option value="<?= $supplier['id_supplier'] ?>" <?= isset($_GET['id_supplier']) && $_GET['id_supplier'] == $supplier['id_supplier'] ? 'selected' : '' ?>>
                                        <?= $supplier['nama_supplier'] ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="id_toko">Toko</label>
                            <select class="form-control" id="id_toko" name="id_toko">
                                <option value="">Semua Toko</option>
                                <?php foreach ($toko_list as $toko): ?>
                                    <option value="<?= $toko['id_toko'] ?>" <?= isset($_GET['id_toko']) && $_GET['id_toko'] == $toko['id_toko'] ? 'selected' : '' ?>>
                                        <?= $toko['nama_toko'] ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label for="status_pembayaran">Status Pembayaran</label>
                            <select class="form-control" id="status_pembayaran" name="status_pembayaran">
                                <option value="">Semua Status</option>
                                <option value="lunas" <?= isset($_GET['status_pembayaran']) && $_GET['status_pembayaran'] == 'lunas' ? 'selected' : '' ?>>Lunas</option>
                                <option value="hutang" <?= isset($_GET['status_pembayaran']) && $_GET['status_pembayaran'] == 'hutang' ? 'selected' : '' ?>>Hutang</option>
                                <option value="sebagian" <?= isset($_GET['status_pembayaran']) && $_GET['status_pembayaran'] == 'sebagian' ? 'selected' : '' ?>>Sebagian</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <div class="d-grid gap-2">
                                <button type="submit" name="filter" class="btn btn-primary">
                                    <i class="fas fa-filter"></i> Filter
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="row mt-2">
                    <div class="col-md-12">
                        <a href="pembelian.php" class="btn btn-secondary">
                            <i class="fas fa-sync"></i> Reset
                        </a>
                        <button type="button" class="btn btn-success float-end" onclick="exportToExcel()">
                            <i class="fas fa-file-excel"></i> Export Excel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Tabel Data Pembelian -->
    <div class="card shadow mb-4 animate-fade-in">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Daftar Transaksi Pembelian</h6>
        </div>
        <div class="card-body">
            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?= $_SESSION['success'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?= $_SESSION['error'] ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['error']); ?>
            <?php endif; ?>
            
            <div class="table-responsive">
                <table class="table table-bordered table-hover" id="dataTable" width="100%" cellspacing="0">
                    <thead class="thead-light">
                        <tr>
                            <th width="5%">No</th>
                            <th width="10%">No Faktur</th>
                            <th width="10%">Tanggal</th>
                            <th width="15%">Supplier</th>
                            <th width="10%">Toko</th>
                            <th width="10%">Metode</th>
                            <th width="10%">Total</th>
                            <th width="10%">Grand Total</th>
                            <th width="10%">Dibayar</th>
                            <th width="10%">Status</th>
                            <th width="10%">Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $no = 1; ?>
                        <?php foreach ($pembelian as $trx): ?>
                            <tr>
                                <td><?= $no++ ?></td>
                                <td><?= $trx['no_faktur'] ?></td>
                                <td><?= date('d-m-Y H:i', strtotime($trx['tanggal_pembelian'])) ?></td>
                                <td><?= $trx['nama_supplier'] ?></td>
                                <td><?= $trx['nama_toko'] ?></td>
                                <td><?= $trx['nama_metode'] ?></td>
                                <td class="text-right"><?= format_rupiah($trx['total_pembelian']) ?></td>
                                <td class="text-right"><?= format_rupiah($trx['grand_total']) ?></td>
                                <td class="text-right"><?= format_rupiah($trx['jumlah_bayar']) ?></td>
                                <td>
                                    <?php if ($trx['status_pembayaran'] == 'lunas'): ?>
                                        <span class="badge bg-success">Lunas</span>
                                    <?php elseif ($trx['status_pembayaran'] == 'hutang'): ?>
                                        <span class="badge bg-danger">Hutang</span>
                                    <?php else: ?>
                                        <span class="badge bg-warning text-dark">Sebagian</span>
                                    <?php endif; ?>
                                </td>
                                <td class="text-center">
                                    <a href="detail_pembelian.php?id=<?= $trx['id_pembelian'] ?>" class="btn btn-sm btn-info" title="Detail">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <?php if ($_SESSION['id_role'] == 1 || $_SESSION['id_role'] == 5): ?>
                                        <a href="pembelian.php?edit=<?= $trx['id_pembelian'] ?>" class="btn btn-sm btn-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="pembelian.php?hapus=<?= $trx['id_pembelian'] ?>" class="btn btn-sm btn-danger" title="Hapus" onclick="return confirm('Apakah Anda yakin ingin menghapus transaksi ini?')">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    <?php endif; ?>
                                    <?php 
                                    // Cek apakah ada nota untuk pembelian ini
                                    $query_nota = "SELECT * FROM nota_pembelian WHERE id_pembelian = ?";
                                    $stmt_nota = mysqli_prepare($conn, $query_nota);
                                    mysqli_stmt_bind_param($stmt_nota, 'i', $trx['id_pembelian']);
                                    mysqli_stmt_execute($stmt_nota);
                                    $result_nota = mysqli_stmt_get_result($stmt_nota);
                                    $nota = mysqli_fetch_assoc($result_nota);
                                    ?>
                                    <?php if ($nota): ?>
                                        <a href="<?= $nota['path_file'] ?>" class="btn btn-sm btn-secondary" title="Lihat Nota" target="_blank">
                                            <i class="fas fa-file-invoice"></i>
                                        </a>
                                    <?php else: ?>
                                        <button class="btn btn-sm btn-secondary" title="Tidak ada nota" disabled>
                                            <i class="fas fa-file-invoice"></i>
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Modal Tambah Pembelian -->
<div class="modal fade" id="tambahPembelianModal" tabindex="-1" aria-labelledby="tambahPembelianModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <form id="formPembelian" method="POST" action="pembelian.php" enctype="multipart/form-data">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="tambahPembelianModalLabel">Tambah Pembelian Baru</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="no_faktur">No Faktur</label>
                                <input type="text" class="form-control" id="no_faktur" name="no_faktur" value="<?= generate_no_transaksi() ?>" readonly>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="tanggal_pembelian">Tanggal Pembelian</label>
                                <input type="datetime-local" class="form-control" id="tanggal_pembelian" name="tanggal_pembelian" value="<?= date('Y-m-d\TH:i') ?>" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="id_toko">Toko</label>
                                <select class="form-control" id="id_toko" name="id_toko" required>
                                    <option value="">Pilih Toko</option>
                                    <?php foreach ($toko_list as $toko): ?>
                                        <option value="<?= $toko['id_toko'] ?>"><?= $toko['nama_toko'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="id_supplier">Supplier</label>
                                <select class="form-control" id="id_supplier" name="id_supplier" required>
                                    <option value="">Pilih Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?= $supplier['id_supplier'] ?>"><?= $supplier['nama_supplier'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="id_metode">Metode Pembayaran</label>
                                <select class="form-control" id="id_metode" name="id_metode" required>
                                    <option value="">Pilih Metode</option>
                                    <?php foreach ($metode_list as $metode): ?>
                                        <option value="<?= $metode['id_metode'] ?>"><?= $metode['nama_metode'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="pencarian_barang">Cari Barang</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="pencarian_barang" placeholder="Masukkan nama atau barcode barang">
                                    <button class="btn btn-outline-secondary" type="button" id="btnCariBarang">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-12">
                            <div class="table-responsive">
                                <table class="table table-bordered" id="tabelBarang">
                                    <thead class="table-light">
                                        <tr>
                                            <th width="25%">Nama Barang</th>
                                            <th width="10%">Jumlah</th>
                                            <th width="15%">Harga Beli</th>
                                            <th width="15%">Diskon</th>
                                            <th width="15%">Expired Date</th>
                                            <th width="20%">Subtotal</th>
                                            <th width="5%">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody id="daftarBarang">
                                        <!-- Barang akan ditambahkan disini -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="catatan">Catatan</label>
                                <textarea class="form-control" id="catatan" name="catatan" rows="3"></textarea>
                            </div>
                            <div class="form-group">
                                <label for="nota">Upload Nota Pembelian</label>
                                <input type="file" class="form-control" id="nota" name="nota" accept="image/*,.pdf">
                                <small class="text-muted">Format: JPG, PNG, PDF (Maks. 2MB)</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-body">
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Total</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="total_pembelian" name="total_pembelian" value="0" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Diskon</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="diskon" name="diskon" value="0">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">PPN</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="ppn" name="ppn" value="0">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Grand Total</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="grand_total" name="grand_total" value="0" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Dibayar</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="jumlah_bayar" name="jumlah_bayar" value="0">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Kembalian</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="kembalian" name="kembalian" value="0" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Status</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-center" id="status_pembayaran" value="Lunas" readonly>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                    <button type="submit" name="tambah_pembelian" class="btn btn-primary">Simpan Transaksi</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Edit Pembelian -->
<?php if ($edit_data): ?>
<div class="modal fade" id="editPembelianModal" tabindex="-1" aria-labelledby="editPembelianModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <form id="formEditPembelian" method="POST" action="pembelian.php" enctype="multipart/form-data">
                <input type="hidden" name="id_pembelian" value="<?= $edit_data['id_pembelian'] ?>">
                <div class="modal-header bg-warning text-white">
                    <h5 class="modal-title" id="editPembelianModalLabel">Edit Pembelian</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="edit_no_faktur">No Faktur</label>
                                <input type="text" class="form-control" id="edit_no_faktur" value="<?= $edit_data['no_faktur'] ?>" readonly>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="edit_tanggal_pembelian">Tanggal Pembelian</label>
                                <input type="datetime-local" class="form-control" id="edit_tanggal_pembelian" name="tanggal_pembelian" value="<?= date('Y-m-d\TH:i', strtotime($edit_data['tanggal_pembelian'])) ?>" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="edit_id_toko">Toko</label>
                                <select class="form-control" id="edit_id_toko" name="id_toko" required>
                                    <option value="">Pilih Toko</option>
                                    <?php foreach ($toko_list as $toko): ?>
                                        <option value="<?= $toko['id_toko'] ?>" <?= $edit_data['id_toko'] == $toko['id_toko'] ? 'selected' : '' ?>><?= $toko['nama_toko'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="edit_id_supplier">Supplier</label>
                                <select class="form-control" id="edit_id_supplier" name="id_supplier" required>
                                    <option value="">Pilih Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?= $supplier['id_supplier'] ?>" <?= $edit_data['id_supplier'] == $supplier['id_supplier'] ? 'selected' : '' ?>><?= $supplier['nama_supplier'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="edit_id_metode">Metode Pembayaran</label>
                                <select class="form-control" id="edit_id_metode" name="id_metode" required>
                                    <option value="">Pilih Metode</option>
                                    <?php foreach ($metode_list as $metode): ?>
                                        <option value="<?= $metode['id_metode'] ?>" <?= $edit_data['id_metode'] == $metode['id_metode'] ? 'selected' : '' ?>><?= $metode['nama_metode'] ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="edit_pencarian_barang">Cari Barang</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="edit_pencarian_barang" placeholder="Masukkan nama atau barcode barang">
                                    <button class="btn btn-outline-secondary" type="button" id="edit_btnCariBarang">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-12">
                            <div class="table-responsive">
                                <table class="table table-bordered" id="edit_tabelBarang">
                                    <thead class="table-light">
                                        <tr>
                                            <th width="25%">Nama Barang</th>
                                            <th width="10%">Jumlah</th>
                                            <th width="15%">Harga Beli</th>
                                            <th width="15%">Diskon</th>
                                            <th width="15%">Expired Date</th>
                                            <th width="20%">Subtotal</th>
                                            <th width="5%">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody id="edit_daftarBarang">
                                        <?php foreach ($edit_detail as $detail): ?>
                                            <tr data-id="<?= $detail['id_barang'] ?>">
                                                <td>
                                                    <?= $detail['nama_barang'] ?> (<?= $detail['kode_barang'] ?>)
                                                    <input type="hidden" name="id_barang[]" value="<?= $detail['id_barang'] ?>">
                                                </td>
                                                <td>
                                                    <input type="number" class="form-control jumlah" name="jumlah[]" value="<?= $detail['jumlah'] ?>" min="1" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                                                </td>
                                                <td>
                                                    <input type="text" class="form-control text-right harga_beli" name="harga_beli[]" value="<?= format_rupiah($detail['harga_beli']) ?>" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                                                </td>
                                                <td>
                                                    <input type="text" class="form-control text-right diskon_item" name="diskon_item[]" value="<?= format_rupiah($detail['diskon_item'] ?? 0) ?>" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                                                </td>
                                                <td>
                                                    <input type="date" class="form-control expired_date" name="expired_date[]" value="<?= $detail['expired_date'] ?>">
                                                </td>
                                                <td>
                                                    <input type="text" class="form-control text-right subtotal" name="subtotal[]" value="<?= format_rupiah($detail['subtotal']) ?>" readonly>
                                                </td>
                                                <td class="text-center">
                                                    <button type="button" class="btn btn-sm btn-danger" onclick="$(this).closest('tr').remove(); hitungTotalEdit()">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="edit_catatan">Catatan</label>
                                <textarea class="form-control" id="edit_catatan" name="catatan" rows="3"><?= $edit_data['catatan'] ?></textarea>
                            </div>
                            <div class="form-group">
                                <label for="edit_nota">Upload Nota Pembelian</label>
                                <input type="file" class="form-control" id="edit_nota" name="nota" accept="image/*,.pdf">
                                <small class="text-muted">Format: JPG, PNG, PDF (Maks. 2MB)</small>
                                <?php 
                                // Cek apakah ada nota untuk pembelian ini
                                $query_nota = "SELECT * FROM nota_pembelian WHERE id_pembelian = ?";
                                $stmt_nota = mysqli_prepare($conn, $query_nota);
                                mysqli_stmt_bind_param($stmt_nota, 'i', $edit_data['id_pembelian']);
                                mysqli_stmt_execute($stmt_nota);
                                $result_nota = mysqli_stmt_get_result($stmt_nota);
                                $nota = mysqli_fetch_assoc($result_nota);
                                ?>
                                <?php if ($nota): ?>
                                    <div class="mt-2">
                                        <a href="<?= $nota['path_file'] ?>" target="_blank">Lihat Nota Saat Ini</a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-body">
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Total</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_total_pembelian" name="total_pembelian" value="<?= format_rupiah($edit_data['total_pembelian']) ?>" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Diskon</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_diskon" name="diskon" value="<?= format_rupiah($edit_data['diskon']) ?>">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">PPN</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_ppn" name="ppn" value="<?= format_rupiah($edit_data['ppn']) ?>">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Grand Total</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_grand_total" name="grand_total" value="<?= format_rupiah($edit_data['grand_total']) ?>" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Dibayar</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_jumlah_bayar" name="jumlah_bayar" value="<?= format_rupiah($edit_data['jumlah_bayar']) ?>">
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Kembalian</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-right" id="edit_kembalian" name="kembalian" value="<?= format_rupiah($edit_data['kembalian']) ?>" readonly>
                                        </div>
                                    </div>
                                    <div class="form-group row">
                                        <label class="col-sm-5 col-form-label">Status</label>
                                        <div class="col-sm-7">
                                            <input type="text" class="form-control text-center" id="edit_status_pembayaran" value="<?= ucfirst($edit_data['status_pembayaran']) ?>" readonly>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                    <button type="submit" name="update_pembelian" class="btn btn-warning">Update Transaksi</button>
                </div>
            </form>
        </div>
    </div>
</div>
<?php endif; ?>

<?php include 'footer.php'; ?>

<script>
// Fungsi untuk format angka dengan separator ribuan
function formatNumber(num) {
    return num.toString().replace(/(\d)(?=(\d{3})+(?!\d))/g, '$1.')
}

// Fungsi untuk menghapus format angka
function unformatNumber(str) {
    return str.replace(/\./g, '')
}

// Fungsi untuk menghitung subtotal per item (tambah pembelian)
function hitungSubtotal(row) {
    const jumlah = parseFloat(unformatNumber(row.find('.jumlah').val())) || 0
    const harga = parseFloat(unformatNumber(row.find('.harga_beli').val())) || 0
    const diskon = parseFloat(unformatNumber(row.find('.diskon_item').val())) || 0
    
    const subtotal = (jumlah * harga) - diskon
    row.find('.subtotal').val(formatNumber(subtotal))
    
    hitungTotal()
}

// Fungsi untuk menghitung total, grand total, kembalian, dan status (tambah pembelian)
function hitungTotal() {
    let total = 0
    
    // Hitung total dari semua subtotal
    $('#daftarBarang tr').each(function() {
        const subtotal = parseFloat(unformatNumber($(this).find('.subtotal').val())) || 0
        total += subtotal
    })
    
    // Update total pembelian
    $('#total_pembelian').val(formatNumber(total))
    
    // Hitung diskon dan ppn
    const diskon = parseFloat(unformatNumber($('#diskon').val())) || 0
    const ppn = parseFloat(unformatNumber($('#ppn').val())) || 0
    
    // Hitung grand total
    const grandTotal = total - diskon + ppn
    $('#grand_total').val(formatNumber(grandTotal))
    
    // Hitung kembalian
    const dibayar = parseFloat(unformatNumber($('#jumlah_bayar').val())) || 0
    const kembalian = dibayar - grandTotal
    $('#kembalian').val(formatNumber(kembalian))
    
    // Update status pembayaran
    if (kembalian >= 0) {
        $('#status_pembayaran').val('Lunas').removeClass('bg-warning bg-danger').addClass('bg-success text-white')
    } else if (dibayar > 0) {
        $('#status_pembayaran').val('Sebagian').removeClass('bg-success bg-danger').addClass('bg-warning text-dark')
    } else {
        $('#status_pembayaran').val('Hutang').removeClass('bg-success bg-warning').addClass('bg-danger text-white')
    }
}

// Fungsi untuk menghitung subtotal per item (edit pembelian)
function hitungSubtotalEdit(row) {
    const jumlah = parseFloat(unformatNumber(row.find('.jumlah').val())) || 0
    const harga = parseFloat(unformatNumber(row.find('.harga_beli').val())) || 0
    const diskon = parseFloat(unformatNumber(row.find('.diskon_item').val())) || 0
    
    const subtotal = (jumlah * harga) - diskon
    row.find('.subtotal').val(formatNumber(subtotal))
    
    hitungTotalEdit()
}

// Fungsi untuk menghitung total, grand total, kembalian, dan status (edit pembelian)
function hitungTotalEdit() {
    let total = 0
    
    // Hitung total dari semua subtotal
    $('#edit_daftarBarang tr').each(function() {
        const subtotal = parseFloat(unformatNumber($(this).find('.subtotal').val())) || 0
        total += subtotal
    })
    
    // Update total pembelian
    $('#edit_total_pembelian').val(formatNumber(total))
    
    // Hitung diskon dan ppn
    const diskon = parseFloat(unformatNumber($('#edit_diskon').val())) || 0
    const ppn = parseFloat(unformatNumber($('#edit_ppn').val())) || 0
    
    // Hitung grand total
    const grandTotal = total - diskon + ppn
    $('#edit_grand_total').val(formatNumber(grandTotal))
    
    // Hitung kembalian
    const dibayar = parseFloat(unformatNumber($('#edit_jumlah_bayar').val())) || 0
    const kembalian = dibayar - grandTotal
    $('#edit_kembalian').val(formatNumber(kembalian))
    
    // Update status pembayaran
    if (kembalian >= 0) {
        $('#edit_status_pembayaran').val('Lunas').removeClass('bg-warning bg-danger').addClass('bg-success text-white')
    } else if (dibayar > 0) {
        $('#edit_status_pembayaran').val('Sebagian').removeClass('bg-success bg-danger').addClass('bg-warning text-dark')
    } else {
        $('#edit_status_pembayaran').val('Hutang').removeClass('bg-success bg-warning').addClass('bg-danger text-white')
    }
}

// Fungsi untuk menambahkan barang ke tabel (tambah pembelian)
function tambahBarang(id_barang, kode_barang, nama_barang, harga_beli) {
    // Cek apakah barang sudah ada di tabel
    if ($(`#daftarBarang tr[data-id="${id_barang}"]`).length > 0) {
        // Jika sudah ada, tambahkan jumlahnya
        const row = $(`#daftarBarang tr[data-id="${id_barang}"]`)
        const jumlah = parseInt(row.find('.jumlah').val()) + 1
        row.find('.jumlah').val(jumlah)
        hitungSubtotal(row)
    } else {
        // Jika belum ada, tambahkan baris baru
        const html = `
            <tr data-id="${id_barang}">
                <td>
                    ${nama_barang} (${kode_barang})
                    <input type="hidden" name="id_barang[]" value="${id_barang}">
                </td>
                <td>
                    <input type="number" class="form-control jumlah" name="jumlah[]" value="1" min="1" onchange="hitungSubtotal($(this).closest('tr'))">
                </td>
                <td>
                    <input type="text" class="form-control text-right harga_beli" name="harga_beli[]" value="${formatNumber(harga_beli)}" onchange="hitungSubtotal($(this).closest('tr'))">
                </td>
                <td>
                    <input type="text" class="form-control text-right diskon_item" name="diskon_item[]" value="0" onchange="hitungSubtotal($(this).closest('tr'))">
                </td>
                <td>
                    <input type="date" class="form-control expired_date" name="expired_date[]">
                </td>
                <td>
                    <input type="text" class="form-control text-right subtotal" name="subtotal[]" value="${formatNumber(harga_beli)}" readonly>
                </td>
                <td class="text-center">
                    <button type="button" class="btn btn-sm btn-danger" onclick="$(this).closest('tr').remove(); hitungTotal()">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `
        $('#daftarBarang').append(html)
        hitungTotal()
    }
    
    // Reset pencarian
    $('#pencarian_barang').val('')
}

// Fungsi untuk menambahkan barang ke tabel (edit pembelian)
function tambahBarangEdit(id_barang, kode_barang, nama_barang, harga_beli) {
    // Cek apakah barang sudah ada di tabel
    if ($(`#edit_daftarBarang tr[data-id="${id_barang}"]`).length > 0) {
        // Jika sudah ada, tambahkan jumlahnya
        const row = $(`#edit_daftarBarang tr[data-id="${id_barang}"]`)
        const jumlah = parseInt(row.find('.jumlah').val()) + 1
        row.find('.jumlah').val(jumlah)
        hitungSubtotalEdit(row)
    } else {
        // Jika belum ada, tambahkan baris baru
        const html = `
            <tr data-id="${id_barang}">
                <td>
                    ${nama_barang} (${kode_barang})
                    <input type="hidden" name="id_barang[]" value="${id_barang}">
                </td>
                <td>
                    <input type="number" class="form-control jumlah" name="jumlah[]" value="1" min="1" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                </td>
                <td>
                    <input type="text" class="form-control text-right harga_beli" name="harga_beli[]" value="${formatNumber(harga_beli)}" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                </td>
                <td>
                    <input type="text" class="form-control text-right diskon_item" name="diskon_item[]" value="0" onchange="hitungSubtotalEdit($(this).closest('tr'))">
                </td>
                <td>
                    <input type="date" class="form-control expired_date" name="expired_date[]">
                </td>
                <td>
                    <input type="text" class="form-control text-right subtotal" name="subtotal[]" value="${formatNumber(harga_beli)}" readonly>
                </td>
                <td class="text-center">
                    <button type="button" class="btn btn-sm btn-danger" onclick="$(this).closest('tr').remove(); hitungTotalEdit()">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `
        $('#edit_daftarBarang').append(html)
        hitungTotalEdit()
    }
    
    // Reset pencarian
    $('#edit_pencarian_barang').val('')
}

$(document).ready(function() {
    // Format input number dengan separator ribuan
    $('body').on('keyup', '.harga_beli, .diskon_item, #diskon, #ppn, #jumlah_bayar, #edit_diskon, #edit_ppn, #edit_jumlah_bayar', function() {
        const value = unformatNumber($(this).val())
        if (!isNaN(value)) {
            $(this).val(formatNumber(value))
        }
    })
    
    // Hitung total ketika ada perubahan pada input (tambah pembelian)
    $('body').on('change', '#diskon, #ppn, #jumlah_bayar', function() {
        hitungTotal()
    })
    
    // Hitung total ketika ada perubahan pada input (edit pembelian)
    $('body').on('change', '#edit_diskon, #edit_ppn, #edit_jumlah_bayar', function() {
        hitungTotalEdit()
    })
    
    // Pencarian barang (tambah pembelian)
    $('#btnCariBarang').click(function() {
        const keyword = $('#pencarian_barang').val()
        if (keyword.length < 2) return
        
        $.ajax({
            url: 'ajax_cari_barang_pembelian.php',
            type: 'GET',
            data: { keyword: keyword },
            dataType: 'json',
            success: function(response) {
                if (response.length > 0) {
                    // Tampilkan hasil pencarian dalam modal
                    let html = '<div class="list-group">'
                    response.forEach(function(barang) {
                        html += `
                            <a href="#" class="list-group-item list-group-item-action" 
                               onclick="tambahBarang(${barang.id_barang}, '${barang.kode_barang}', '${barang.nama_barang}', ${barang.harga_beli}); return false;">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1">${barang.nama_barang}</h6>
                                    <small>${barang.kode_barang}</small>
                                </div>
                                <p class="mb-1">Harga: Rp ${formatNumber(barang.harga_beli)} | Stok: ${barang.stok}</p>
                                <small>${barang.nama_kategori}</small>
                            </a>
                        `
                    })
                    html += '</div>'
                    
                    // Tampilkan modal hasil pencarian
                    $('#modalHasilPencarian .modal-body').html(html)
                    $('#modalHasilPencarian').modal('show')
                } else {
                    alert('Barang tidak ditemukan')
                }
            }
        })
    })
    
    // Pencarian barang (edit pembelian)
    $('#edit_btnCariBarang').click(function() {
        const keyword = $('#edit_pencarian_barang').val()
        if (keyword.length < 2) return
        
        $.ajax({
            url: 'ajax_cari_barang_pembelian.php',
            type: 'GET',
            data: { keyword: keyword },
            dataType: 'json',
            success: function(response) {
                if (response.length > 0) {
                    // Tampilkan hasil pencarian dalam modal
                    let html = '<div class="list-group">'
                    response.forEach(function(barang) {
                        html += `
                            <a href="#" class="list-group-item list-group-item-action" 
                               onclick="tambahBarangEdit(${barang.id_barang}, '${barang.kode_barang}', '${barang.nama_barang}', ${barang.harga_beli}); return false;">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1">${barang.nama_barang}</h6>
                                    <small>${barang.kode_barang}</small>
                                </div>
                                <p class="mb-1">Harga: Rp ${formatNumber(barang.harga_beli)} | Stok: ${barang.stok}</p>
                                <small>${barang.nama_kategori}</small>
                            </a>
                        `
                    })
                    html += '</div>'
                    
                    // Tampilkan modal hasil pencarian
                    $('#modalHasilPencarianEdit .modal-body').html(html)
                    $('#modalHasilPencarianEdit').modal('show')
                } else {
                    alert('Barang tidak ditemukan')
                }
            }
        })
    })
    
    // Event listener untuk enter pada input pencarian (tambah pembelian)
    $('#pencarian_barang').keypress(function(e) {
        if (e.which == 13) {
            e.preventDefault()
            $('#btnCariBarang').click()
        }
    })
    
    // Event listener untuk enter pada input pencarian (edit pembelian)
    $('#edit_pencarian_barang').keypress(function(e) {
        if (e.which == 13) {
            e.preventDefault()
            $('#edit_btnCariBarang').click()
        }
    })
    
    // Buka modal edit jika ada parameter edit di URL
    <?php if (isset($_GET['edit'])): ?>
        $('#editPembelianModal').modal('show')
        
        // Hitung total awal untuk edit
        hitungTotalEdit()
    <?php endif; ?>
})

// Fungsi untuk export ke Excel
function exportToExcel() {
    // Dapatkan parameter filter dari URL
    const params = new URLSearchParams(window.location.search)
    
    // Redirect ke halaman export dengan parameter yang sama
    window.location.href = 'export_pembelian_excel.php?' + params.toString()
}
</script>

<!-- Modal Hasil Pencarian Barang (Tambah Pembelian) -->
<div class="modal fade" id="modalHasilPencarian" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Hasil Pencarian Barang</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <!-- Hasil pencarian akan ditampilkan disini -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal Hasil Pencarian Barang (Edit Pembelian) -->
<div class="modal fade" id="modalHasilPencarianEdit" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Hasil Pencarian Barang</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <!-- Hasil pencarian akan ditampilkan disini -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>