<?php 
// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

include 'header.php';
require_once 'config.php';

// Periksa apakah pengguna sudah login
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Inisialisasi variabel
$errors = [];
$success = false;

// Ambil data pengguna yang sedang login
$id_pengguna = $_SESSION['id_pengguna'];
$query = "SELECT * FROM pengguna WHERE id_pengguna = ?";
$stmt = mysqli_prepare($conn, $query);
mysqli_stmt_bind_param($stmt, "i", $id_pengguna);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$user_data = mysqli_fetch_assoc($result);

// Proses form jika ada data yang dikirim
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validasi input
    $nama_lengkap = trim($_POST['nama_lengkap']);
    $username = trim($_POST['username']);
    $current_password = trim($_POST['current_password']);
    $new_password = trim($_POST['new_password']);
    $confirm_password = trim($_POST['confirm_password']);
    
    // Validasi nama lengkap
    if (empty($nama_lengkap)) {
        $errors['nama_lengkap'] = 'Nama lengkap harus diisi';
    } elseif (strlen($nama_lengkap) < 3) {
        $errors['nama_lengkap'] = 'Nama lengkap minimal 3 karakter';
    }
    
    // Validasi username
    if (empty($username)) {
        $errors['username'] = 'Username harus diisi';
    } elseif (strlen($username) < 3) {
        $errors['username'] = 'Username minimal 3 karakter';
    } else {
        // Cek apakah username sudah digunakan oleh pengguna lain
        $query = "SELECT id_pengguna FROM pengguna WHERE username = ? AND id_pengguna != ?";
        $stmt = mysqli_prepare($conn, $query);
        mysqli_stmt_bind_param($stmt, "si", $username, $id_pengguna);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) > 0) {
            $errors['username'] = 'Username sudah digunakan';
        }
    }
    
    // Validasi password jika diisi
    $password_changed = false;
    if (!empty($current_password) || !empty($new_password) || !empty($confirm_password)) {
        // Verifikasi password saat ini
        if (!password_verify($current_password, $user_data['password'])) {
            $errors['current_password'] = 'Password saat ini salah';
        }
        
        // Validasi password baru
        if (empty($new_password)) {
            $errors['new_password'] = 'Password baru harus diisi';
        } elseif (strlen($new_password) < 6) {
            $errors['new_password'] = 'Password minimal 6 karakter';
        }
        
        // Validasi konfirmasi password
        if ($new_password !== $confirm_password) {
            $errors['confirm_password'] = 'Konfirmasi password tidak cocok';
        }
        
        $password_changed = true;
    }
    
    // Proses upload foto profil jika ada
    $foto_profil = $user_data['foto_profil'];
    if (isset($_FILES['foto_profil']) && $_FILES['foto_profil']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['foto_profil'];
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $allowed_ext = ['jpg', 'jpeg', 'png', 'gif'];
        
        if (!in_array(strtolower($ext), $allowed_ext)) {
            $errors['foto_profil'] = 'Format file tidak didukung (hanya JPG, PNG, GIF)';
        } elseif ($file['size'] > 2 * 1024 * 1024) { // 2MB
            $errors['foto_profil'] = 'Ukuran file terlalu besar (maksimal 2MB)';
        } else {
            // Buat direktori jika belum ada
            if (!is_dir('uploads/profil')) {
                mkdir('uploads/profil', 0777, true);
            }
            
            // Hapus foto lama jika ada
            if ($foto_profil && file_exists('uploads/profil/' . $foto_profil)) {
                unlink('uploads/profil/' . $foto_profil);
            }
            
            // Generate nama file unik
            $foto_profil = 'profile_' . $id_pengguna . '_' . time() . '.' . $ext;
            move_uploaded_file($file['tmp_name'], 'uploads/profil/' . $foto_profil);
        }
    }
    
    // Jika tidak ada error, update data
    if (empty($errors)) {
        if ($password_changed) {
            // Update dengan password baru
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $query = "UPDATE pengguna SET nama_lengkap = ?, username = ?, password = ?, foto_profil = ? WHERE id_pengguna = ?";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "ssssi", $nama_lengkap, $username, $hashed_password, $foto_profil, $id_pengguna);
        } else {
            // Update tanpa password
            $query = "UPDATE pengguna SET nama_lengkap = ?, username = ?, foto_profil = ? WHERE id_pengguna = ?";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "sssi", $nama_lengkap, $username, $foto_profil, $id_pengguna);
        }
        
        if (mysqli_stmt_execute($stmt)) {
            $success = true;
            
            // Update session jika username berubah
            if ($_SESSION['username'] !== $username) {
                $_SESSION['username'] = $username;
            }
            
            // Refresh data pengguna
            $query = "SELECT * FROM pengguna WHERE id_pengguna = ?";
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "i", $id_pengguna);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);
            $user_data = mysqli_fetch_assoc($result);
        } else {
            $errors['general'] = 'Gagal menyimpan perubahan: ' . mysqli_error($conn);
        }
    }
}
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Pengaturan Akun</h1>
        <a href="profile.php" class="d-none d-sm-inline-block btn btn-sm btn-primary-custom">
            <i class="fas fa-user fa-sm"></i> Kembali ke Profil
        </a>
    </div>
    
    <!-- Notifikasi -->
    <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show animate-fade-in" role="alert" style="animation-delay: 0.1s;">
            <strong>Berhasil!</strong> Perubahan akun telah disimpan.
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php elseif (isset($errors['general'])): ?>
        <div class="alert alert-danger alert-dismissible fade show animate-fade-in" role="alert" style="animation-delay: 0.1s;">
            <strong>Error!</strong> <?php echo $errors['general']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4 animate-fade-in" style="animation-delay: 0.2s;">
                <div class="card-header py-3 bg-success text-white">
                    <h6 class="m-0 font-weight-bold">Informasi Dasar</h6>
                </div>
                <div class="card-body">
                    <form method="POST" enctype="multipart/form-data">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="nama_lengkap" class="form-label">Nama Lengkap</label>
                                <input type="text" class="form-control <?php echo isset($errors['nama_lengkap']) ? 'is-invalid' : ''; ?>" 
                                       id="nama_lengkap" name="nama_lengkap" 
                                       value="<?php echo htmlspecialchars($user_data['nama_lengkap']); ?>">
                                <?php if (isset($errors['nama_lengkap'])): ?>
                                    <div class="invalid-feedback"><?php echo $errors['nama_lengkap']; ?></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="col-md-6">
                                <label for="username" class="form-label">Username</label>
                                <input type="text" class="form-control <?php echo isset($errors['username']) ? 'is-invalid' : ''; ?>" 
                                       id="username" name="username" 
                                       value="<?php echo htmlspecialchars($user_data['username']); ?>">
                                <?php if (isset($errors['username'])): ?>
                                    <div class="invalid-feedback"><?php echo $errors['username']; ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" 
                                   value="<?php echo htmlspecialchars($user_data['email']); ?>" disabled>
                            <small class="text-muted">Hubungi admin untuk mengubah email</small>
                        </div>
                        
                        <hr class="my-4">
                        
                        <h5 class="mb-3">Ubah Password</h5>
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label for="current_password" class="form-label">Password Saat Ini</label>
                                <input type="password" class="form-control <?php echo isset($errors['current_password']) ? 'is-invalid' : ''; ?>" 
                                       id="current_password" name="current_password">
                                <?php if (isset($errors['current_password'])): ?>
                                    <div class="invalid-feedback"><?php echo $errors['current_password']; ?></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="col-md-4">
                                <label for="new_password" class="form-label">Password Baru</label>
                                <input type="password" class="form-control <?php echo isset($errors['new_password']) ? 'is-invalid' : ''; ?>" 
                                       id="new_password" name="new_password">
                                <?php if (isset($errors['new_password'])): ?>
                                    <div class="invalid-feedback"><?php echo $errors['new_password']; ?></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="col-md-4">
                                <label for="confirm_password" class="form-label">Konfirmasi Password</label>
                                <input type="password" class="form-control <?php echo isset($errors['confirm_password']) ? 'is-invalid' : ''; ?>" 
                                       id="confirm_password" name="confirm_password">
                                <?php if (isset($errors['confirm_password'])): ?>
                                    <div class="invalid-feedback"><?php echo $errors['confirm_password']; ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <small class="text-muted">Biarkan kosong jika tidak ingin mengubah password</small>
                        </div>
                        
                        <hr class="my-4">
                        
                        <div class="mb-3">
                            <label for="foto_profil" class="form-label">Foto Profil</label>
                            <input type="file" class="form-control <?php echo isset($errors['foto_profil']) ? 'is-invalid' : ''; ?>" 
                                   id="foto_profil" name="foto_profil" accept="image/*">
                            <?php if (isset($errors['foto_profil'])): ?>
                                <div class="invalid-feedback"><?php echo $errors['foto_profil']; ?></div>
                            <?php endif; ?>
                            <small class="text-muted">Format: JPG, PNG, GIF (maks. 2MB)</small>
                        </div>
                        
                        <?php if ($user_data['foto_profil']): ?>
                            <div class="mb-3">
                                <img src="uploads/profil/<?php echo htmlspecialchars($user_data['foto_profil']); ?>" 
                                     class="img-thumbnail" style="width: 100px; height: 100px; object-fit: cover;" 
                                     alt="Foto Profil Saat Ini">
                                <div class="form-check mt-2">
                                    <input class="form-check-input" type="checkbox" id="hapus_foto" name="hapus_foto">
                                    <label class="form-check-label" for="hapus_foto">
                                        Hapus foto profil
                                    </label>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="d-flex justify-content-end mt-4">
                            <button type="submit" class="btn btn-primary-custom">
                                <i class="fas fa-save me-1"></i> Simpan Perubahan
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-4">
            <div class="card shadow mb-4 animate-fade-in" style="animation-delay: 0.3s;">
                <div class="card-header py-3 bg-success text-white">
                    <h6 class="m-0 font-weight-bold">Panduan Keamanan</h6>
                </div>
                <div class="card-body">
                    <div class="alert alert-warning">
                        <h6 class="alert-heading"><i class="fas fa-exclamation-triangle me-1"></i> Tips Keamanan</h6>
                        <ul class="small">
                            <li>Gunakan password yang kuat dan unik</li>
                            <li>Jangan bagikan password Anda dengan siapapun</li>
                            <li>Ganti password secara berkala</li>
                            <li>Pastikan alamat email Anda valid</li>
                        </ul>
                    </div>
                    
                    <div class="alert alert-info">
                        <h6 class="alert-heading"><i class="fas fa-info-circle me-1"></i> Informasi</h6>
                        <p class="small">Beberapa informasi seperti email dan role tidak dapat diubah sendiri. Hubungi administrator sistem untuk perubahan tersebut.</p>
                    </div>
                </div>
            </div>
            
            <div class="card shadow animate-fade-in" style="animation-delay: 0.4s;">
                <div class="card-header py-3 bg-danger text-white">
                    <h6 class="m-0 font-weight-bold">Zona Berbahaya</h6>
                </div>
                <div class="card-body">
                    <button class="btn btn-outline-danger w-100 mb-2" data-bs-toggle="modal" data-bs-target="#logoutModal">
                        <i class="fas fa-sign-out-alt me-1"></i> Logout
                    </button>
                    
                    <button class="btn btn-outline-danger w-100" data-bs-toggle="modal" data-bs-target="#deactivateModal">
                        <i class="fas fa-user-slash me-1"></i> Nonaktifkan Akun
                    </button>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Logout Modal -->
<div class="modal fade" id="logoutModal" tabindex="-1" aria-labelledby="logoutModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title" id="logoutModalLabel">Konfirmasi Logout</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                Apakah Anda yakin ingin logout dari sistem?
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <a href="logout.php" class="btn btn-primary-custom">Ya, Logout</a>
            </div>
        </div>
    </div>
</div>

<!-- Deactivate Modal -->
<div class="modal fade" id="deactivateModal" tabindex="-1" aria-labelledby="deactivateModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="deactivateModalLabel">Nonaktifkan Akun</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Dengan menonaktifkan akun, Anda tidak akan bisa login sampai administrator mengaktifkannya kembali.</p>
                <p class="fw-bold">Apakah Anda yakin ingin menonaktifkan akun?</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-danger" disabled>
                    <i class="fas fa-user-slash me-1"></i> Nonaktifkan Akun
                </button>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>