<?php
// Start session dan validasi login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Cek apakah user sudah login dan memiliki akses
if (!isset($_SESSION['id_pengguna'])) {
    header("Location: login.php");
    exit();
}

// Validasi hak akses (hanya admin dan marketing)
$allowed_roles = ['0', '1', '2', '3', '4'];
if (!isset($_SESSION['id_role']) || !in_array($_SESSION['id_role'], $allowed_roles)) {
    header("Location: unauthorized.php");
    exit();
}

include 'header.php';
require_once 'config.php';

// Fungsi untuk format mata uang
function format_rupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

// Query untuk statistik promo
$query_statistik = "SELECT 
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as aktif,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as nonaktif,
                    COUNT(*) as total
                    FROM promo";
$result_statistik = mysqli_query($conn, $query_statistik);
$statistik = mysqli_fetch_assoc($result_statistik);

// Query untuk data promo dengan pencarian
$search = isset($_GET['search']) ? $_GET['search'] : '';
$query = "SELECT p.*, t.nama_toko 
          FROM promo p
          LEFT JOIN toko t ON p.id_toko = t.id_toko
          WHERE (p.nama_promo LIKE ? OR p.kode_promo LIKE ?)";

// Gunakan prepared statement untuk pencarian
$stmt = mysqli_prepare($conn, $query);
$search_param = "%$search%";
mysqli_stmt_bind_param($stmt, "ss", $search_param, $search_param);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

// Konfigurasi pagination
$per_page = 10;
$total_records = mysqli_num_rows($result);
$total_pages = ceil($total_records / $per_page);
$current_page = isset($_GET['page']) ? $_GET['page'] : 1;
$offset = ($current_page - 1) * $per_page;

// Query dengan pagination
$query_pagination = "SELECT p.*, t.nama_toko 
                    FROM promo p
                    LEFT JOIN toko t ON p.id_toko = t.id_toko
                    WHERE (p.nama_promo LIKE ? OR p.kode_promo LIKE ?)
                    ORDER BY p.created_at DESC
                    LIMIT ?, ?";
$stmt_pagination = mysqli_prepare($conn, $query_pagination);
$limit_param = $per_page;
$offset_param = $offset;
mysqli_stmt_bind_param($stmt_pagination, "ssii", $search_param, $search_param, $offset_param, $limit_param);
mysqli_stmt_execute($stmt_pagination);
$result_pagination = mysqli_stmt_get_result($stmt_pagination);
?>

<!-- Main Content -->
<main class="animate-fade-in">
    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Manajemen Promo</h1>
        <div class="d-none d-sm-inline-block">
            <span class="text-muted">Terakhir diperbarui: <?php echo date('d M Y H:i:s'); ?></span>
        </div>
    </div>

    <!-- Statistik Ringkasan -->
    <div class="row mb-4">
        <!-- Total Promo -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total Promo</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['total']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-tags fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Promo Aktif -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Promo Aktif</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['aktif']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Promo Nonaktif -->
        <div class="col-xl-4 col-md-6 mb-4">
            <div class="card border-left-danger shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                Promo Nonaktif</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $statistik['nonaktif']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-times-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Card untuk tabel -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between bg-primary-custom">
            <h6 class="m-0 font-weight-bold text-white">Daftar Promo</h6>
            <div>
                <a href="tambah_promo.php" class="btn btn-sm btn-success">
                    <i class="fas fa-plus"></i> Tambah Promo Baru
                </a>
            </div>
        </div>
        <div class="card-body">
            <!-- Form Pencarian -->
            <form method="GET" action="" class="mb-4">
                <div class="input-group">
                    <input type="text" class="form-control" name="search" placeholder="Cari berdasarkan nama atau kode promo..." value="<?php echo htmlspecialchars($search); ?>">
                    <button class="btn btn-primary-custom" type="submit">
                        <i class="fas fa-search"></i> Cari
                    </button>
                    <?php if (!empty($search)): ?>
                        <a href="promo.php" class="btn btn-danger">
                            <i class="fas fa-times"></i> Reset
                        </a>
                    <?php endif; ?>
                </div>
            </form>

            <?php if (mysqli_num_rows($result_pagination) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" id="dataTable" width="100%" cellspacing="0">
                        <thead class="bg-light">
                            <tr>
                                <th>No</th>
                                <th>Kode Promo</th>
                                <th>Nama Promo</th>
                                <th>Jenis</th>
                                <th>Nilai</th>
                                <th>Periode</th>
                                <th>Toko</th>
                                <th>Status</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $no = 1 + $offset;
                            while ($row = mysqli_fetch_assoc($result_pagination)): 
                                // Format nilai promo
                                $nilai_promo = ($row['tipe_nilai'] == 'persen') ? 
                                    $row['nilai_promo'] . '%' : 
                                    format_rupiah($row['nilai_promo']);
                                
                                // Format periode
                                $tanggal_mulai = date('d M Y', strtotime($row['tanggal_mulai']));
                                $tanggal_selesai = date('d M Y', strtotime($row['tanggal_selesai']));
                                $periode = "$tanggal_mulai - $tanggal_selesai";
                                
                                // Warna badge jenis promo
                                $badge_class = '';
                                switch ($row['jenis_promo']) {
                                    case 'diskon': $badge_class = 'bg-info'; break;
                                    case 'gratis': $badge_class = 'bg-success'; break;
                                    case 'bundling': $badge_class = 'bg-warning text-dark'; break;
                                }
                            ?>
                                <tr>
                                    <td><?php echo $no++; ?></td>
                                    <td><?php echo htmlspecialchars($row['kode_promo']); ?></td>
                                    <td><?php echo htmlspecialchars($row['nama_promo']); ?></td>
                                    <td>
                                        <span class="badge <?php echo $badge_class; ?>">
                                            <?php echo ucfirst($row['jenis_promo']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo $nilai_promo; ?></td>
                                    <td><?php echo $periode; ?></td>
                                    <td><?php echo $row['nama_toko'] ? htmlspecialchars($row['nama_toko']) : 'Semua Toko'; ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $row['is_active'] ? 'success' : 'danger'; ?>">
                                            <?php echo $row['is_active'] ? 'Aktif' : 'Nonaktif'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-info btn-barang" 
                                                data-id="<?php echo $row['id_promo']; ?>"
                                                title="Lihat Barang Promo">
                                            <i class="fas fa-boxes"></i>
                                        </button>
                                        <a href="detail_promo.php?id=<?php echo $row['id_promo']; ?>" 
                                           class="btn btn-sm btn-primary" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="edit_promo.php?id=<?php echo $row['id_promo']; ?>" 
                                           class="btn btn-sm btn-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <button class="btn btn-sm btn-danger btn-hapus" 
                                                data-id="<?php echo $row['id_promo']; ?>"
                                                data-nama="<?php echo htmlspecialchars($row['nama_promo']); ?>"
                                                title="Hapus">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <?php if ($current_page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $current_page - 1; ?>&search=<?php echo urlencode($search); ?>" aria-label="Previous">
                                    <span aria-hidden="true">&laquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($i == $current_page) ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($current_page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $current_page + 1; ?>&search=<?php echo urlencode($search); ?>" aria-label="Next">
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php else: ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Tidak ada data promo yang ditemukan.
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<!-- Modal untuk menampilkan daftar barang promo -->
<div class="modal fade" id="barangModal" tabindex="-1" role="dialog" aria-labelledby="barangModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary-custom text-white">
                <h5 class="modal-title" id="barangModalLabel">Daftar Barang Promo</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div class="table-responsive">
                    <table class="table table-bordered" id="tableBarang">
                        <thead>
                            <tr>
                                <th>No</th>
                                <th>Kode Barang</th>
                                <th>Nama Barang</th>
                            </tr>
                        </thead>
                        <tbody id="barangBody">
                            <!-- Data akan diisi via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal konfirmasi hapus -->
<div class="modal fade" id="hapusModal" tabindex="-1" role="dialog" aria-labelledby="hapusModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="hapusModalLabel">Konfirmasi Hapus Promo</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Anda yakin ingin menghapus promo <strong id="namaPromoHapus"></strong>?</p>
                <p class="text-danger">Aksi ini tidak dapat dibatalkan!</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Batal</button>
                <form id="formHapus" method="POST" action="hapus_promo.php">
                    <input type="hidden" name="id_promo" id="idPromoHapus">
                    <button type="submit" class="btn btn-danger">Hapus</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Include DataTables CSS & JS -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap4.min.css">
<script type="text/javascript" src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
<script type="text/javascript" src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap4.min.js"></script>

<script>
$(document).ready(function() {
    // Inisialisasi DataTables
    $('#dataTable').DataTable({
        "paging": false, // Nonaktifkan pagination bawaan DataTables
        "searching": false, // Nonaktifkan search bawaan DataTables
        "info": false, // Nonaktifkan info bawaan DataTables
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.11.5/i18n/Indonesian.json"
        }
    });

    // Modal untuk menampilkan daftar barang promo
    $('.btn-barang').click(function() {
        var id_promo = $(this).data('id');
        
        // Kirim request AJAX untuk mendapatkan daftar barang
        $.ajax({
            url: 'get_barang_promo.php',
            method: 'GET',
            data: {id_promo: id_promo},
            dataType: 'json',
            success: function(response) {
                var html = '';
                if (response.length > 0) {
                    $.each(response, function(index, barang) {
                        html += '<tr>' +
                                '<td>' + (index + 1) + '</td>' +
                                '<td>' + barang.kode_barang + '</td>' +
                                '<td>' + barang.nama_barang + '</td>' +
                                '</tr>';
                    });
                } else {
                    html = '<tr><td colspan="3" class="text-center">Tidak ada barang dalam promo ini</td></tr>';
                }
                
                $('#barangBody').html(html);
                $('#barangModal').modal('show');
            },
            error: function() {
                alert('Terjadi kesalahan saat memuat data barang');
            }
        });
    });

    // Modal konfirmasi hapus
    $('.btn-hapus').click(function() {
        var id = $(this).data('id');
        var nama = $(this).data('nama');
        
        $('#namaPromoHapus').text(nama);
        $('#idPromoHapus').val(id);
        $('#hapusModal').modal('show');
    });
});
</script>

<?php include 'footer.php'; ?>