<?php
// File: register.php
// Deskripsi: Halaman registrasi untuk sistem toko kuliner

require_once 'config.php';

// Redirect jika sudah login
if (isset($_SESSION['user_id'])) {
    redirect('dashboard.php');
}

$error = '';
$success = '';

// Proses registrasi
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['register'])) {
    $nama_lengkap = clean_input($_POST['nama_lengkap']);
    $username = clean_input($_POST['username']);
    $email = clean_input($_POST['email']);
    $password = clean_input($_POST['password']);
    $password_confirm = clean_input($_POST['password_confirm']);
    $telepon = clean_input($_POST['telepon']);

    // Validasi input
    if (empty($nama_lengkap) || empty($username) || empty($email) || empty($password) || empty($password_confirm)) {
        $error = "Semua field harus diisi!";
    } elseif ($password !== $password_confirm) {
        $error = "Password tidak cocok!";
    } elseif (strlen($password) < 6) {
        $error = "Password minimal 6 karakter!";
    } else {
        // Cek apakah username sudah ada
        $stmt = $conn->prepare("SELECT id_pengguna FROM pengguna WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $stmt->store_result();
        
        if ($stmt->num_rows > 0) {
            $error = "Username sudah digunakan!";
        } else {
            // Cek apakah email sudah ada
            $stmt = $conn->prepare("SELECT id_pengguna FROM pengguna WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $stmt->store_result();
            
            if ($stmt->num_rows > 0) {
                $error = "Email sudah digunakan!";
            } else {
                // Hash password
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                // Default role untuk user baru (2 = Kasir)
                $default_role = 2;
                
                // Insert user baru
                $stmt = $conn->prepare("INSERT INTO pengguna (username, password, nama_lengkap, email, telepon, id_role) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->bind_param("sssssi", $username, $hashed_password, $nama_lengkap, $email, $telepon, $default_role);
                
                if ($stmt->execute()) {
                    redirect('login.php?register=success');
                } else {
                    $error = "Gagal membuat akun. Silakan coba lagi!";
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Sistem Manajemen SPOG STORE</title>
    
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary: #2e7d32; /* Hijau */
            --primary-dark: #1b5e20; /* Hijau tua */
            --secondary: #d32f2f; /* Merah */
            --light: #f8f9fa;
            --dark: #212121;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            height: 100vh;
            display: flex;
            align-items: center;
            background: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), 
                        url('assets/bg.jpg') no-repeat center center fixed;
            background-size: cover;
        }
        
        .register-container {
            max-width: 500px;
            width: 100%;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
            overflow: hidden;
        }
        
        .register-header {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
            padding: 25px;
            text-align: center;
        }
        
        .register-header h2 {
            margin: 0;
            font-weight: 700;
            font-size: 1.8rem;
        }
        
        .register-body {
            padding: 30px;
        }
        
        .btn-register {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            height: 50px;
            border-radius: 8px;
            font-weight: 600;
            border: none;
            width: 100%;
        }
        
        .btn-register:hover {
            background: linear-gradient(to right, var(--primary-dark), var(--primary));
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="register-container">
            <div class="register-header">
                <h2>DAFTAR AKUN BARU</h2>
                <p>Sistem Manajemen SPOG STORE</p>
            </div>
            
            <div class="register-body">
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-circle me-2"></i> <?= $error ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <div class="mb-3">
                        <label for="nama_lengkap" class="form-label">Nama Lengkap</label>
                        <input type="text" class="form-control" id="nama_lengkap" name="nama_lengkap" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">Email</label>
                        <input type="email" class="form-control" id="email" name="email" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="telepon" class="form-label">Nomor Telepon</label>
                        <input type="tel" class="form-control" id="telepon" name="telepon" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" class="form-control" id="password" name="password" required>
                    </div>
                    
                    <div class="mb-4">
                        <label for="password_confirm" class="form-label">Konfirmasi Password</label>
                        <input type="password" class="form-control" id="password_confirm" name="password_confirm" required>
                    </div>
                    
                    <button type="submit" name="register" class="btn btn-register mb-3">
                        <i class="fas fa-user-plus me-2"></i> DAFTAR
                    </button>
                    
                    <div class="text-center pt-3 border-top">
                        <p class="mb-0">Sudah punya akun? <a href="login.php" class="fw-bold">Login disini</a></p>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>